/****************************************************************************
*
* Copyright © 2021 STMicroelectronics - All Rights Reserved
*
* This software is licensed under SLA0098 terms that can be found in the
* DM00779817_1_0.pdf file in the licenses directory of this software product.
*
* THIS SOFTWARE IS DISTRIBUTED "AS IS," AND ALL WARRANTIES ARE DISCLAIMED,
* INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*
*****************************************************************************/
/**
 * @file    sdmmc_lld.h
 * @brief   SPC5xx SDMMC control unit.
 *
 * @addtogroup SDMMC
 * @{
 */

#include "sdmmc_lld.h"

#if (LLD_USE_SDMMC == TRUE) || defined(__DOXYGEN__)

#include "sdmmc_lld_cfg.h"
#include "string.h"

/* SDMMC Terms and definitions
 * HD = Host driver
 * HC = Host controller
 * SDMMC = MMC specification version 4.51 and SD HC Standard Specification Version 3.0.
 */
typedef enum {
	SDMMC_CMD0_GO_IDLE_STATE = 0,
	SDMMC_CMD1_SEND_OP_COND,
	SDMMC_CMD2_ALL_SEND_CID,
	SDMMC_CMD3_SET_RELATIVE_ADDR,
	SDMMC_CMD4_SET_DSR,
	SDMMC_CMD5_SLEEP_AWAKE,
	SDMMC_CMD6_SWITCH,
	SDMMC_CMD7_SELECT_CARD,
	SDMMC_CMD8_SEND_EXT_CSD,
	SDMMC_CMD9_SEND_CSD,
	SDMMC_CMD10_SEND_CID,
	SDMMC_CMD11_OBSOLETE,
	SDMMC_CMD12_STOP_TRANSMISSION,
	SDMMC_CMD13_SEND_STATUS,
	SDMMC_CMD14_BUSTEST_R,
	SDMMC_CMD15_GO_INACTIVE_STATE,
	SDMMC_CMD16_SET_BLOCKLEN,
	SDMMC_CMD17_READ_SINGLE_BLOCK,
	SDMMC_CMD18_READ_MULTIPLE_BLOCK,
	SDMMC_CMD19_BUSTEST_W,
	SDMMC_CMD21_SEND_TUNING_BLOCK = 21,
	SDMMC_CMD23_SET_BLOCK_COUNT = 23,
	SDMMC_CMD24_WRITE_BLOCK,
	SDMMC_CMD25_WRITE_MULTIPLE_BLOCK,
	SDMMC_CMD26_PROGRAM_CID,
	SDMMC_CMD27_PROGRAM_CSD,
	SDMMC_CMD28_SET_WRITE_PRO,
	SDMMC_CMD29_CLR_WRITE_PROT,
	SDMMC_CMD30_SEND_WRITE_PRO_T,
	SDMMC_CMD31_SEND_WRITE_PRO_T_TYPE,
	SDMMC_CMD35_ERASE_GROUP_STAR_T = 35,
	SDMMC_CMD36_ERASE_GROUP_END,
	SDMMC_CMD38_ERASE = 38,
	SDMMC_CMD39_FAST_IO,
	SDMMC_CMD40_GO_IRQ_STATE,
	SDMMC_CMD42_LOCK_UNLOCK = 42,
	SDMMC_CMD55_APP_CMD = 55,
	SDMMC_CMD56_GEN_CMD,
} sdmmc_cmd_t;

typedef enum {
	SDMMC_NO_RESPONSE,	// code length 0 bits
	SDMMC_R2_CID_CSD,	// code length 136 bits
	SDMMC_R1_NORMAL,	// code length 48 bits
	SDMMC_R1B_BUSY,		// code length 48 bits
	SDMMC_R3_OCR,		// code length 48 bits
	SDMMC_R4_FAST_IO,	// code length 48 bits
	SDMMC_R5_IRQ,		// code length 48 bits
} sdmmc_response_t;

typedef enum {
	HD_STATE_IDLE,
	HD_STATE_READY,
	HD_STATE_IDENT,
	HD_STATE_STBY,
	HD_STATE_TRAN,
	HD_STATE_DATA,
	HD_STATE_RCV,
	HD_STATE_PRG,
	HD_STATE_DIS,
	HD_STATE_BTST,
	HD_STATE_SLP,
	HD_ERROR_SWITCH = 0x10,
	HD_ERROR_ERASE,
	HD_ERROR_CSD,
	HD_ERROR_DEVICE,
	HD_ERROR_LOCK,
	HD_ERROR_BLOCKLEN,
	HD_ERROR_ADDRESS
} hd_state_t;

typedef enum {
	SDMMC_MODE_CMD_SET,
	SDMMC_MODE_SET_BITS,
	SDMMC_MODE_CLEAR_BITS,
	SDMMC_MODE_WRITE_BYTE
} sdmmc_mode_csd_t;

// Host Controller definition
typedef enum {
	HC_TYPE_NORMAL,
	HC_TYPE_SUSPEND,
	HC_TYPE_RESUME,
	HC_TYPE_ABORT,
} hc_cmd_type_t;

typedef enum {
	HC_STATE_DAT,
	HC_STATE_CMD,
	HC_STATE_CARD,
	HC_STATE_READ,
	HC_STATE_WRITE,
	HC_STATE_CLOCK,
	HC_STATE_BOOT,
} hc_state_t;

typedef enum {
	HC_LEVEL_SIGNAL,
	HC_LEVEL_STATUS
} hc_level_t;

typedef enum {
	HC_CLOCK_POWER = 400000,
	HC_CLOCK_DEFAULT = 25000000,
	HC_CLOCK_HIGH = 50000000,
	HC_CLOCK_HS200 = 200000000,
	HC_CLOCK_HS400 = 400000000,
} hc_clock_t;

// Card Device Identification
sdmmc_card_id_t card_id;
// Card Device Operation
sdmmc_card_op_t card_op;
// Card State Machine
hd_state_t card_sm;
// Card Extended Property
uint8_t card_ext[512];

/*******************************************************************************
* Local function prototypes
*******************************************************************************/
static sdmmc_result_t sdmmc_send_cmd(sdmmc_cmd_t cmd, uint32_t arg, uint32_t nbytes, uint16_t timeout);
static void sdmmc_reset(hc_state_t state);
static inline float sdmmc_multiple_factors(uint8_t m) {
	float multiple_factors[] = {0.0, 1.0, 1.2, 1.3, 1.5, 2.0, 2.6, 3.0,
								   3.5, 4.0, 4.5, 5.2, 5.5, 6.0, 7.0, 8.0};
	return multiple_factors[m];
}

static uint32_t sdmmc_convert_speed(uint32_t speed) {
	uint32_t frequency[] = {100, 1000, 10000, 100000};
	return (frequency[speed & 0x3] * sdmmc_multiple_factors((speed & 0x78) >> 3) * 1000);
}

static hc_cmd_type_t sdmmc_command_type(sdmmc_cmd_t cmd, uint32_t arg) {
	uint16_t arg16 = (uint16_t)arg;

	if ((cmd == SDMMC_CMD5_SLEEP_AWAKE) && (arg16 != 0))
		return HC_TYPE_SUSPEND;
	if ((cmd == SDMMC_CMD5_SLEEP_AWAKE) && (arg16 == 0))
		return HC_TYPE_RESUME;
	return HC_TYPE_NORMAL;
}

static sdmmc_response_t sdmmc_response_type(sdmmc_cmd_t cmd) {
	switch (cmd) {
	case SDMMC_CMD0_GO_IDLE_STATE:
	case SDMMC_CMD4_SET_DSR:
	case SDMMC_CMD15_GO_INACTIVE_STATE:
		return SDMMC_NO_RESPONSE;
	case SDMMC_CMD3_SET_RELATIVE_ADDR:
	case SDMMC_CMD7_SELECT_CARD:
	case SDMMC_CMD8_SEND_EXT_CSD:
	case SDMMC_CMD12_STOP_TRANSMISSION:
	case SDMMC_CMD13_SEND_STATUS:
	case SDMMC_CMD14_BUSTEST_R:
	case SDMMC_CMD19_BUSTEST_W:
	case SDMMC_CMD16_SET_BLOCKLEN:
	case SDMMC_CMD17_READ_SINGLE_BLOCK:
	case SDMMC_CMD18_READ_MULTIPLE_BLOCK:
	case SDMMC_CMD21_SEND_TUNING_BLOCK:
	case SDMMC_CMD23_SET_BLOCK_COUNT:
	case SDMMC_CMD24_WRITE_BLOCK:
	case SDMMC_CMD25_WRITE_MULTIPLE_BLOCK:
	case SDMMC_CMD26_PROGRAM_CID:
	case SDMMC_CMD27_PROGRAM_CSD:
	case SDMMC_CMD30_SEND_WRITE_PRO_T:
	case SDMMC_CMD31_SEND_WRITE_PRO_T_TYPE:
	case SDMMC_CMD35_ERASE_GROUP_STAR_T:
	case SDMMC_CMD36_ERASE_GROUP_END:
	case SDMMC_CMD42_LOCK_UNLOCK:
	case SDMMC_CMD55_APP_CMD:
	case SDMMC_CMD56_GEN_CMD:
		return SDMMC_R1_NORMAL;
	case SDMMC_CMD5_SLEEP_AWAKE:
	case SDMMC_CMD6_SWITCH:
	case SDMMC_CMD28_SET_WRITE_PRO:
	case SDMMC_CMD29_CLR_WRITE_PROT:
	case SDMMC_CMD38_ERASE:
		return SDMMC_R1B_BUSY;
	case SDMMC_CMD2_ALL_SEND_CID:
	case SDMMC_CMD9_SEND_CSD:
	case SDMMC_CMD10_SEND_CID:
		return SDMMC_R2_CID_CSD;
	case SDMMC_CMD1_SEND_OP_COND:
		return SDMMC_R3_OCR;
	case SDMMC_CMD39_FAST_IO:
		return SDMMC_R4_FAST_IO;
	case SDMMC_CMD40_GO_IRQ_STATE:
		return SDMMC_R5_IRQ;
	default:
		return SDMMC_NO_RESPONSE;
	}
}

static sdmmc_response_t sdmmc_response_length(sdmmc_cmd_t cmd) {
	/* Mapping between response type register
	 * and MMC response specification
	 */
	switch (sdmmc_response_type(cmd)) {
	case SDMMC_NO_RESPONSE:
		return SDMMC_NO_RESPONSE;
	case SDMMC_R2_CID_CSD:
		return SDMMC_R2_CID_CSD;
	case SDMMC_R1B_BUSY:
		return SDMMC_R1B_BUSY;
	default:
		return SDMMC_R1_NORMAL;
	}
}

static uint8_t sdmmc_cmd_direction(sdmmc_cmd_t cmd) {
	switch (cmd) {
	case SDMMC_CMD0_GO_IDLE_STATE:
	case SDMMC_CMD1_SEND_OP_COND:
	case SDMMC_CMD2_ALL_SEND_CID:
	case SDMMC_CMD8_SEND_EXT_CSD:
	case SDMMC_CMD13_SEND_STATUS:
	case SDMMC_CMD14_BUSTEST_R:
	case SDMMC_CMD17_READ_SINGLE_BLOCK:
	case SDMMC_CMD18_READ_MULTIPLE_BLOCK:
		// 1 Read (card to Host)
		return 0x1;
	default:
		// 0 Write (Host to card)
		return 0x0;
	}
}

static uint16_t sdmmc_cmd_support(sdmmc_cmd_t cmd) {
	// Class 0 basic always supported
	if ((cmd >= SDMMC_CMD0_GO_IDLE_STATE) &&
		(cmd <= SDMMC_CMD10_SEND_CID))
		return 0x1;
	if ((cmd >= SDMMC_CMD12_STOP_TRANSMISSION) &&
		(cmd <= SDMMC_CMD15_GO_INACTIVE_STATE))
		return 0x1;
	if (cmd == SDMMC_CMD19_BUSTEST_W)
		return 0x1;
	// Class 2 block read
	if ((cmd >= SDMMC_CMD16_SET_BLOCKLEN) &&
		(cmd <= SDMMC_CMD18_READ_MULTIPLE_BLOCK))
		return card_op.command_class & 0x4;
	// Class 4 block write
	if ((cmd >= SDMMC_CMD23_SET_BLOCK_COUNT) &&
		(cmd <= SDMMC_CMD27_PROGRAM_CSD))
		return card_op.command_class & 0x10;
	// Class 5 erase
	if ((cmd >= SDMMC_CMD35_ERASE_GROUP_STAR_T) &&
		(cmd <= SDMMC_CMD38_ERASE))
		return card_op.command_class & 0x20;
	// Class 6 write protection
	if ((cmd >= SDMMC_CMD28_SET_WRITE_PRO) &&
		(cmd <= SDMMC_CMD31_SEND_WRITE_PRO_T_TYPE))
		return card_op.command_class & 0x20;
	// Class 7 lock device
	if (cmd == SDMMC_CMD42_LOCK_UNLOCK)
		return card_op.command_class & 0x80;
	// Class 8 specific application
	if ((cmd >= SDMMC_CMD55_APP_CMD) &&
		(cmd <= SDMMC_CMD56_GEN_CMD))
		return card_op.command_class & 0x100;
	// Class 9 I/O mode
	if ((cmd >= SDMMC_CMD39_FAST_IO) &&
		(cmd <= SDMMC_CMD40_GO_IRQ_STATE))
		return card_op.command_class & 0x200;
	return 0x0;
}


static hd_state_t sdmmc_get_device_state(void) {
	uint32_t status;

	// This is the real current state of device as response of commands
	// identifications process. We can move to STBY and verify that is 
	// into a valid state.
	if (card_sm == HD_STATE_IDENT)
		card_sm = HD_STATE_STBY;

	// The valid states where can be performed a command status request
	if ((card_sm >= HD_STATE_STBY) && (card_sm <= HD_STATE_BTST)) {
		sdmmc_send_cmd(SDMMC_CMD13_SEND_STATUS,
						card_op.device_rca, 0x0, 0xfff);
	}

	status = (SDMMC.RESPONSE[1].R << 16) | SDMMC.RESPONSE[0].R;
	if (status & 0x00000080UL) return HD_ERROR_SWITCH;
	if (status & 0x1800a000UL) return HD_ERROR_ERASE;
	if (status & 0x00010000UL) return HD_ERROR_CSD;
	if (status & 0x00f80000UL) return HD_ERROR_DEVICE;
	if (status & 0x07000000UL) return HD_ERROR_LOCK;
	if (status & 0x20000000UL) return HD_ERROR_BLOCKLEN;
	if (status & 0xc0000000UL) return HD_ERROR_ADDRESS;
	card_sm = (status & 0x1e00) >> 0x9;
	return card_sm;
}

static void sdmmc_switching_state(uint32_t rca, hd_state_t req_sm) {
	sdmmc_result_t result = SDMMC_REQUEST_ERROR;

	// Move to Transfer State if required
	while (card_sm != req_sm) {
		if (card_sm >= HD_ERROR_SWITCH)
			return;
		if (card_sm == HD_STATE_DATA) {
			// Move from DATA to STBY
			result = sdmmc_send_cmd(SDMMC_CMD7_SELECT_CARD,
									rca, 0x0, 0xff);
		}
		if (card_sm == HD_STATE_STBY) {
			// Move from STBY to TRAN
			if (req_sm == HD_STATE_TRAN)
				result = sdmmc_send_cmd(SDMMC_CMD7_SELECT_CARD,
										rca, 0x0, 0xfff);
			// Move from STBY to SLEEP
			if (req_sm == HD_STATE_SLP) {
				// From now only an awake or a device reset commands
				// are accepted.
				result = sdmmc_send_cmd(SDMMC_CMD5_SLEEP_AWAKE,
										rca | (0x1UL << 15), 0x0,
										card_op.sa_timeout);
				if (result == SDMMC_NO_ERROR)
					card_sm = HD_STATE_SLP;
				return;
			}
		}
		if (card_sm == HD_STATE_TRAN) {
			// Move from TRAN to STBY.
			result = sdmmc_send_cmd(SDMMC_CMD7_SELECT_CARD,
									0x0, 0x0, 0xfff);
		}
		if (card_sm == HD_STATE_SLP) {
			// Move from SLP to STBY (AWAKE)
			result = sdmmc_send_cmd(SDMMC_CMD5_SLEEP_AWAKE,
									rca | (0x0UL << 15), 0x0,
									card_op.sa_timeout);
		}
		if (result == SDMMC_NO_ERROR)
			card_sm = sdmmc_get_device_state();
		else
			return;
	}
}

static inline void sdmmc_clear_error(void) {
	SDMMC.ERRORINTRSTS.R = 0x137f;
	SDMMC.NORMALINTRSTS.R = 0x50ff;

	// Reset CMD/DAT lines in case remains in inhibit state for next CMD/DAT
	if ((SDMMC.PRESENTSTATE.R & 0x1) == 0x1)
		sdmmc_reset(HC_STATE_CMD);
	if ((SDMMC.PRESENTSTATE.R & 0x2) == 0x2)
		sdmmc_reset(HC_STATE_DAT);
}

static uint16_t sdmmc_is_active_state(hc_level_t level, hc_state_t state) {
	uint16_t is_active = 0x0;

	if (HC_STATE_DAT == state) {
		if (level == HC_LEVEL_SIGNAL) {
			is_active = SDMMC.PRESENTSTATE.R & 0x006;
		} else {
			is_active = SDMMC.NORMALINTRSTS.R & 0x2;
		}
	}
	if (HC_STATE_CMD == state) {
		if (level == HC_LEVEL_SIGNAL) {
			is_active = SDMMC.PRESENTSTATE.R & 0x1;
		} else {
			is_active = SDMMC.NORMALINTRSTS.R & 0x1;
		}
	}
	if (HC_STATE_CARD == state) {
		if (level == HC_LEVEL_SIGNAL) {
			is_active = (SDMMC.PRESENTSTATE.R & 0x00050000) >> 16;
		} else {
			is_active = (SDMMC.NORMALINTRSTS.R & 0x0040) ||	SDMMC.SLOTINTRSTS.R;
		}
	}
	if (HC_STATE_READ == state) {
		if (level == HC_LEVEL_SIGNAL) {
			is_active = SDMMC.PRESENTSTATE.R & 0x206;
		} else {
			is_active = SDMMC.NORMALINTRSTS.R & 0x2;
		}
	}
	if (HC_STATE_WRITE == state) {
		if (level == HC_LEVEL_SIGNAL) {
			is_active = SDMMC.PRESENTSTATE.R & 0x106;
		} else {
			is_active = SDMMC.NORMALINTRSTS.R & 0x2;
		}
	}
	if (HC_STATE_CLOCK == state)
		is_active = SDMMC.CLOCKCONTROL.R & 0x2;

	return is_active;
}

static sdmmc_result_t sdmmc_wait_hd(hd_state_t state, uint16_t mdelay) {
	uint32_t btime = osalThreadGetMilliseconds();

	// 0xffff is a infinite wait.
	while (sdmmc_get_device_state() != state) {
		if (mdelay < (osalThreadGetMilliseconds() - btime)) {
			if (mdelay != 0xffff)
				return SDMMC_TIMEOUT_ERROR;
		}
	}
	return SDMMC_NO_ERROR;
}

static sdmmc_result_t sdmmc_wait_hc(hc_level_t level, hc_state_t state, uint16_t mdelay) {
	uint32_t btime = osalThreadGetMilliseconds();

	// 0xffff is a infinite wait.
	while (sdmmc_is_active_state(level, state) == 0x0) {
		if (mdelay < (osalThreadGetMilliseconds() - btime)) {
			if (mdelay != 0xffff)
				return SDMMC_TIMEOUT_ERROR;
		}
	}
	return SDMMC_NO_ERROR;
}

/* 5.4Extended CSD Register
 * The Extended CSD register defines the Device properties and selected modes.
 * It is 512 bytes long. The most significant 320 bytes are the Properties segment,
 * which defines the Device capabilities and cannot be modified by the host.
 * The lower 192 bytes are the Modes segment, which defines the configuration the
 * Device is working in. These modes can be changed by the host by means of the SWITCH command.
*/
static sdmmc_result_t sdmmc_config_device_property(uint16_t reg, uint8_t val, uint8_t op) {
	sdmmc_result_t result;
	uint32_t value;

	if (card_sm != HD_STATE_TRAN)
		return SDMMC_SM_ERROR;

	if (op == 0x0) {
		// 0 Write (Host to card)
		value = SDMMC_MODE_WRITE_BYTE << 24; // write byte mode
		value|= ((uint32_t)reg << 16); // index
		value|= ((uint32_t)val << 8); // value
		result = sdmmc_send_cmd(SDMMC_CMD6_SWITCH,
								value, 0x0, card_op.cmd6_timeout);
		sdmmc_wait_hd(HD_STATE_TRAN, 0xfff);
	} else {
		// 1 Read (card to Host)
		PCM_0.PCM3.B.EMMC_MSTR_EN_BYTE_SWAP = 0x0;
		SDMMC.SDMASYSADDR.R = (uint32_t)&card_ext[0];
		result = sdmmc_send_cmd(SDMMC_CMD8_SEND_EXT_CSD,
								0x0, card_op.read_block_length,
								card_op.data_timeout);
		PCM_0.PCM3.B.EMMC_MSTR_EN_BYTE_SWAP = 0x1;
	}
	return result;
}

static sdmmc_result_t sdmmc_send_cmd(sdmmc_cmd_t cmd, uint32_t arg, uint32_t nbytes, uint16_t timeout) {
	sdmmc_result_t result = SDMMC_CMD_ERROR;
	hc_state_t check_state;
	uint16_t reg16;

	// Check cmd is supported
	if (sdmmc_cmd_support(cmd) == 0x0)
		return result;

	// Check if CMD or DAT are running
	if ((sdmmc_is_active_state(HC_LEVEL_SIGNAL, HC_STATE_CMD) != 0x0) ||
		(sdmmc_is_active_state(HC_LEVEL_SIGNAL, HC_STATE_DAT) != 0x0))
		return SDMMC_FATAL_ERROR;

	// Set default values for many commands
	SDMMC.BLOCKSIZE.R = 0x0;
	SDMMC.BLOCKCOUNT.R = 0x0;

	// set block size and block count register
	switch (cmd) {
		// Data transfer or DAT lines check
	case SDMMC_CMD14_BUSTEST_R:
	case SDMMC_CMD17_READ_SINGLE_BLOCK:
	case SDMMC_CMD18_READ_MULTIPLE_BLOCK:
	case SDMMC_CMD8_SEND_EXT_CSD:
		SDMMC.BLOCKSIZE.R = card_op.read_block_length;
		SDMMC.BLOCKCOUNT.R = nbytes / card_op.read_block_length;
		check_state = HC_STATE_READ;
		break;
	case SDMMC_CMD19_BUSTEST_W:
	case SDMMC_CMD24_WRITE_BLOCK:
	case SDMMC_CMD25_WRITE_MULTIPLE_BLOCK:
		SDMMC.BLOCKSIZE.R = card_op.write_block_length;
		SDMMC.BLOCKCOUNT.R = nbytes / card_op.write_block_length;
		check_state = HC_STATE_WRITE;
		break;
	case SDMMC_CMD5_SLEEP_AWAKE:
	case SDMMC_CMD6_SWITCH:
	case SDMMC_CMD38_ERASE:
		// No Data transfer but DAT[0] line busy
		check_state = HC_STATE_DAT;
		break;
	default:
		// No data transfer
		check_state = HC_STATE_CMD;
	}

	// set argument flag
	SDMMC.ARGUMENT1.R = arg;

	// set transfer mode register
	reg16 = (sdmmc_cmd_direction(cmd) << 0x4);
	if (SDMMC.BLOCKCOUNT.B.XFER_BLOCKCOUNT > 0x1)
		reg16|= 0x20; // Set Multiblock
	if (SDMMC.BLOCKCOUNT.B.XFER_BLOCKCOUNT != 0x0)
		reg16|= 0x3; // Set DMA transfer
	SDMMC.TRANSFERMODE.R = reg16;

	/* set command register
	 * Type Index CRC Name
	 * 00 0 0 No response
	 * 01 0 1 R2
	 * 10 0 0 R3, R4
	 * 10 1 1 R1, R6, R5, R7
	 * 11 1 1 R1b, R5b */
	reg16 = sdmmc_response_length(cmd);
	// Enable CRC and Index checks.
	switch (sdmmc_response_type(cmd)) {
	case SDMMC_NO_RESPONSE:
	case SDMMC_R3_OCR:
	case SDMMC_R4_FAST_IO:
		reg16 |= (0x0 << 0x3) | (0x0 << 0x4);
		break;
	case SDMMC_R2_CID_CSD:
		reg16 |= (0x1 << 0x3) | (0x0 << 0x4);
		break;
	case SDMMC_R5_IRQ:
	case SDMMC_R1_NORMAL:
		if ((cmd == SDMMC_CMD7_SELECT_CARD) && (arg == 0x0))
			reg16 = 0x0;
		else
			reg16 |= (0x1 << 0x3) | (0x1 << 0x4);
		break;
	default:
		reg16 |= (0x1 << 0x3) | (0x1 << 0x4);
	}
	// Set CMD type
	reg16 |= (sdmmc_command_type(cmd, arg) << 0x6);
	// Set DAT requests
	if ((check_state == HC_STATE_WRITE) || (check_state == HC_STATE_READ))
		reg16 |= (0x1 << 0x5);
	// Send command
	reg16 |= (cmd << 0x8);
	SDMMC.COMMAND.R = reg16;

	if (SDMMC.NORMALINTRSTS.R & 0x8000) {
		// clear error interrupt command status
		sdmmc_clear_error();
	} else {
		// wait for completion
		result = sdmmc_wait_hc(HC_LEVEL_STATUS, check_state, timeout);
		// clear interrupt status
		SDMMC.NORMALINTRSTS.R = SDMMC.NORMALINTRSTS.R;
		// wait for lines releasing
		if (SDMMC.PRESENTSTATE.R & 0x3)
			result = sdmmc_wait_hc(HC_LEVEL_SIGNAL, check_state, 0xff);
	}

	return result;
}

static sdmmc_result_t sdmmc_set_clock(uint32_t card_clk) {
	uint16_t divider = 0x1;
	uint32_t base_clk;
	uint16_t reg16;

	if (card_clk < HC_CLOCK_POWER)
		return SDMMC_CLOCK_ERROR;

	base_clk = 1000000 * ((SDMMC.CAPABILITIES31_0.R & 0xff00) >> 0x8);
	while ((base_clk / divider / 2) > card_clk) {
		divider++;
	}
	// Disable clock before to change
	SDMMC.CLOCKCONTROL.R = 0x0;
	reg16 = (divider << 0x8) | 0x5;
	// Setup and enable clock
	SDMMC.CLOCKCONTROL.R = reg16;
	// Wait for clock stabilization
	if (sdmmc_wait_hc(HC_LEVEL_SIGNAL, HC_STATE_CLOCK, 0xff) != SDMMC_NO_ERROR)
		return SDMMC_CLOCK_ERROR;
	return SDMMC_NO_ERROR;
}

static sdmmc_result_t sdmmc_get_id(void) {
	sdmmc_result_t result = SDMMC_IDENTIFY_ERROR;
	uint8_t retries = 10;
	uint16_t *pnm;

	do {
		if (sdmmc_send_cmd(SDMMC_CMD2_ALL_SEND_CID,
							0x0, 0x0, 0xfff) == SDMMC_NO_ERROR) {
			/* The CRC7 checksum byte is not reported into response as specified
			 * into the reference manual because used by Host Controller.
			 * R2 (CID, CSD, register) CID or CSD reg. incl. R[127:8] REP[119:0]
			 */
			card_id.manufactur_id = SDMMC.RESPONSE[7].R & 0x00ff;
			card_id.device_type = (SDMMC.RESPONSE[6].R & 0xff00) >> 8;
			card_id.optional_id = SDMMC.RESPONSE[6].R & 0x00ff;
			pnm = (uint16_t*)&card_id.product_name;
			pnm[0] = SDMMC.RESPONSE[5].R;
			pnm[1] = SDMMC.RESPONSE[4].R;
			pnm[2] = SDMMC.RESPONSE[3].R;
			card_id.product_version = (SDMMC.RESPONSE[2].R & 0xff00) >> 8;
			card_id.product_serial_no = (SDMMC.RESPONSE[2].R & 0x00ff) << 24;
			card_id.product_serial_no|= ((uint32_t)SDMMC.RESPONSE[1].R << 8);
			card_id.product_serial_no|= ((SDMMC.RESPONSE[0].R & 0xff00) >> 8);
			card_id.manufactur_data = SDMMC.RESPONSE[0].R & 0x00ff;
			card_op.device_rca = card_id.optional_id << 16;
			card_sm = HD_STATE_IDENT;
			result = SDMMC_NO_ERROR;
			break;
		}
	} while (--retries != 0);
	return result;
}

static sdmmc_result_t sdmmc_get_op(void) {
	uint16_t device_multiple;
	sdmmc_result_t result = SDMMC_CARD_ERROR;
	uint8_t retries = 3;

	// As per specification we should stay in identification state
	if (card_sm == HD_STATE_IDENT)
		do {
			result = sdmmc_send_cmd(SDMMC_CMD3_SET_RELATIVE_ADDR,
									card_op.device_rca, 0x0, 0xff);
		} while ((result != SDMMC_NO_ERROR) && (retries--));
	else
		return result;

	// From now we can use response or command status to know the real device state
	if (result == SDMMC_NO_ERROR)
		sdmmc_get_device_state();

	// The state machine should be stand-by
	if ((result == SDMMC_NO_ERROR) && (card_sm == HD_STATE_STBY)) {
		result = sdmmc_send_cmd(SDMMC_CMD9_SEND_CSD,
								card_op.device_rca, 0x0, 0xfff);
		if (result == SDMMC_NO_ERROR) {
			card_op.device_version = (SDMMC.RESPONSE[7].R & 0x003c) >> 2;
			card_op.data_timeout = (SDMMC.RESPONSE[6].R & 0x7800) >> 11;
			card_op.data_timeout = sdmmc_multiple_factors(card_op.data_timeout) * 20;
			card_op.max_speed = (SDMMC.RESPONSE[5].R & 0xff00) >> 8;
			card_op.max_speed = sdmmc_convert_speed(card_op.max_speed);
			card_op.command_class = ((SDMMC.RESPONSE[5].R & 0x00ff) << 4) |
									(SDMMC.RESPONSE[4].R & 0xf000) >> 12;

			// The read block length is 2^READ_BL_LEN
			card_op.read_block_length = (SDMMC.RESPONSE[4].R & 0x0f00) >> 8;
			card_op.read_block_length = 1 << card_op.read_block_length;

			// Memory capacity = BLOCKNR * BLOCK_LEN
			// BLOCK_LEN = 2^READ_BL_LEN
			// BLOCKNR = (C_SIZE+1) * (2^ (C_SIZE_MULT+2))
			card_op.device_size = ((SDMMC.RESPONSE[3].R & 0xfff0) >> 4) + 1;
			device_multiple = ((SDMMC.RESPONSE[2].R & 0x0380) >> 7) + 2;
			device_multiple = 1 << device_multiple;
			card_op.device_size = card_op.device_size * device_multiple * card_op.read_block_length;
			card_op.device_size = card_op.device_size / 1024 / 1024 / 1024;

			// erasable unit = (ERASE_GRP_SIZE + 1) * (ERASE_GRP_MULT + 1)
			card_op.erase_size = (SDMMC.RESPONSE[2].R & 0x007c) >> 2;
			device_multiple = (SDMMC.RESPONSE[2].R & 0x0003) << 3 |
					(SDMMC.RESPONSE[1].R & 0xe000) >> 13;
			card_op.erase_size = ( card_op.erase_size + 1) * (device_multiple + 1);

			// The write block length is 2^WRITE_BL_LEN
			card_op.write_block_length = (SDMMC.RESPONSE[1].R & 0x0003) << 2 |
					(SDMMC.RESPONSE[0].R & 0xc000) >> 14;
			card_op.write_block_length = 1 << card_op.write_block_length;
			result = SDMMC_NO_ERROR;
		}
	} else {
		result = SDMMC_CARD_ERROR;
		memset(&card_id, 0x0, sizeof(card_id));
		memset(&card_op, 0x0, sizeof(card_op));
		sdmmc_clear_error();
	}
	return result;
}

static sdmmc_result_t sdmmc_card_initialization(void) {
	sdmmc_result_t result = SDMMC_BOOT_ERROR;
	uint8_t retries;

	// Device identification sequence
	retries = 3;
	do {
	    // Send CMD0 to reset the bus, keep CS line high during this step.
		result = sdmmc_send_cmd(SDMMC_CMD0_GO_IDLE_STATE,
								0x0, 0x0, 0xfff);
		if ((result != SDMMC_NO_ERROR) || (card_sm != HD_STATE_IDLE)) {
			// Force reset because not in Pre-Idle state. Stop boot.
			if (result != SDMMC_NO_ERROR) {
				result = sdmmc_send_cmd(SDMMC_CMD0_GO_IDLE_STATE,
										0xFFFFFFFAU, 0x0, 0xfff);
				osalThreadDelayMilliseconds(200);
			}
		}
		// Wait at least 1ms, wait for 74 clock cycles
	    osalThreadDelayMilliseconds(2);
		if (result == SDMMC_NO_ERROR)
			break;
	} while (--retries);
	if (retries == 0x0)
		return result;

	retries = 10;
	do {
		// If the OCR busy bit is 0 steps will be repeated until timeout
		sdmmc_send_cmd(SDMMC_CMD1_SEND_OP_COND,
						0x40ff8000, 0x0, 0xff);
		if ((SDMMC.RESPONSE[1].R & 0x8000) != 0x0) {
			// Check if the Device is a High Voltage or Dual Voltage Device.
			uint32_t response = (SDMMC.RESPONSE[1].R << 16) | SDMMC.RESPONSE[0].R;
			if ((response & 0xff8080) == 0xff8080)
				card_id.dual_voltage = 0x1;
			card_sm = HD_STATE_READY;
			break;
		}
	    osalThreadDelayMilliseconds(2);
	} while (--retries);
	if (retries == 0x0)
		return result;

	// Retrieve Device Identification
	result = sdmmc_get_id();
	if (result == SDMMC_NO_ERROR) {
		// Retrieve Device Operation
		result = sdmmc_get_op();
	}
	return result;
}

/**
 * @brief				Host Controller Software reset.
 *						These registers are RW auto clear. The Host driver requests a
 *						Host Controller operation by setting the bits.
 *						The Host Controller shall clear the bits automatically when
 *						the operation is completed.
 * @param[in] state		CMD or DAT lines reset.
 *
 * @return				No error.
 *
 * @noapi
 */
static void sdmmc_reset(hc_state_t state) {
	switch (state) {
	case HC_STATE_DAT:
		SDMMC.SOFTWARERESET.B.SWRESET_FOR_DAT = 0x1;
		while ((SDMMC.SOFTWARERESET.R & 0x4) != 0x0) {}
		break;
	case HC_STATE_CMD:
		SDMMC.SOFTWARERESET.B.SWRESET_FOR_CMD = 0x1;
		while ((SDMMC.SOFTWARERESET.R & 0x2) != 0x0) {}
		break;
	default:
		SDMMC.SOFTWARERESET.R = 0x1;
		while (SDMMC.SOFTWARERESET.R != 0x0) {}
	}
	return;
}


/*******************************************************************************
* Global functions
*******************************************************************************/
/**
 * @brief				Try to set data bus speed.
 *						If host wants to operate a device with a clock above 26MHz it shall
 *						switch the timing mode to one of supported higher speed timing modes:
 *						High Speed for frequencies between 26MHz and 52MHz and HS200 for
 *						frequencies above 52MHz up to 200MHz (High Speed and HS200 timing
 *						modes allow operation from 0 to 52MHz and 0 to 200MHz.
 * @param[in] speed		One of the configurable sdmmc_speed_t type.
 *
 * @return				No error if the request can be satisfied, sdmmc_result_t error otherwise.
 *
 * @api
 */
sdmmc_result_t sdmmc_switching_speed(sdmmc_speed_t speed) {
	sdmmc_result_t result = SDMMC_REQUEST_ERROR;
	uint32_t reg32, hs_timing, value = 0x0;
	uint8_t device_type = 0x0;

	// Switching to high-speed mode
	if ((card_op.device_version < 0x4) ||
		(speed > SDMMC_SPEED_HS200) ||
		(card_id.manufactur_id == 0x0))
		return result;

	// Check if HC support the request
	if ((SDMMC.CAPABILITIES31_0.B.CORECFG_HIGHSPEEDSUPPORT == 0x1) &&
		(speed > SDMMC_SPEED_HIGH))
		return result;

	sdmmc_switching_state(card_op.device_rca, HD_STATE_TRAN);

	if (card_sm == HD_STATE_TRAN) {
		result = sdmmc_config_device_property(0x0, 0x0, 0x1);
		if (result != SDMMC_NO_ERROR)
			return result;

		// HPI_FEATURES [503]
		card_op.hpi_support = card_ext[503];
		// BUS_WIDTH [183]
		card_op.bus_width = card_ext[183];
		// DEVICE_TYPE [196]
		device_type = card_ext[196];
		// POWER_CLASS [187]
		card_op.power_class = card_ext[187];
		// CMD6_TIMEOUT [248] Time is expressed in units of 10-milliseconds.
		card_op.cmd6_timeout = 10 * card_ext[248];
		// S_A_TIMEOUT [217]
		card_op.sa_timeout = (1 << card_ext[217]) / 10;

		if ((SDMMC_SPEED_LEGACY == speed) && (device_type & 0x01))
			hs_timing = HC_CLOCK_DEFAULT;
		if ((SDMMC_SPEED_HIGH == speed) && (device_type & 0x06))
			hs_timing = HC_CLOCK_HIGH;
		if ((SDMMC_SPEED_HS200 == speed) && (device_type & 0x30))
			hs_timing = HC_CLOCK_HS200;
		if (hs_timing == 0x0)
			return SDMMC_REQUEST_ERROR;

		// HS_TIMING [185]
		result = sdmmc_config_device_property(185, speed, 0x0);
		if (result != SDMMC_NO_ERROR)
			return result;

		// Density in GiB == SECTOR COUNT [215:212] * BLK_LEN
		// The least significant byte (LSB) of the sector count value is the byte [212]
		reg32 = card_ext[215] << 24 |
				card_ext[214] << 16 |
				card_ext[213] << 8  |
				card_ext[212];
		value = (1UL << 30) / card_op.read_block_length;
		card_op.device_size += (reg32 / value);

		// Move from PRG to STBY
		if (card_sm == HD_STATE_PRG)
			sdmmc_switching_state(card_op.device_rca, HD_STATE_TRAN);

		result = sdmmc_config_device_property(0x0, 0x0, 0x1);
		if (result != SDMMC_NO_ERROR)
			return result;

		if (card_ext[185] != speed)
			return SDMMC_REQUEST_ERROR;

		return sdmmc_set_clock(hs_timing);
	}
	return SDMMC_SM_ERROR;
}

sdmmc_result_t sdmmc_changing_data_bus(sdmmc_bus_t bus) {
	sdmmc_result_t result = SDMMC_SM_ERROR;

	// Check if all is satisfied.
	// Device complies with version 4.0, or higher
	if ((card_id.manufactur_id == 0x0) ||
		(bus > SDMMC_BUS_8BIT_SINGLE) ||
		(card_op.device_version < 0x4))
		return SDMMC_REQUEST_ERROR;

	// Check if HC support the request
	if ((SDMMC.CAPABILITIES31_0.B.CORECFG_8BITSUPPORT == 0x0) &&
		(bus == SDMMC_BUS_8BIT_SINGLE))
		return SDMMC_REQUEST_ERROR;

	// Check if already stays into this request
	if (bus == card_op.bus_width)
		return SDMMC_NO_ERROR;

	sdmmc_switching_state(card_op.device_rca, HD_STATE_TRAN);

	if (card_sm == HD_STATE_TRAN) {
		memset(&card_ext, 0x0, card_op.write_block_length);
		/* For 8 data lines the data block would be (MSB to LSB): 0x0000_0000_0000_AA55
		 * For 4 data lines the data block would be (MSB to LSB): 0x0000_005A
		 * For only 1 data line the data block would be: 0x80 */
		switch (bus) {
		case SDMMC_BUS_1BIT_SINGLE:
			card_ext[0] = 0x80;
			SDMMC.HOSTCONTROL1.B.HOSTCTRL1_DATAWIDTH = 0x0;
			SDMMC.HOSTCONTROL1.B.HOSTCTRL1_EXTDATAWIDTH = 0x0;
			break;
		case SDMMC_BUS_4BIT_SINGLE:
			card_ext[0] = 0x5A;
			SDMMC.HOSTCONTROL1.B.HOSTCTRL1_DATAWIDTH = 0x1;
			SDMMC.HOSTCONTROL1.B.HOSTCTRL1_EXTDATAWIDTH = 0x0;
			break;
		case SDMMC_BUS_8BIT_SINGLE:
			card_ext[0] = 0x55;
			card_ext[1] = 0xAA;
			SDMMC.HOSTCONTROL1.B.HOSTCTRL1_DATAWIDTH = 0x0;
			SDMMC.HOSTCONTROL1.B.HOSTCTRL1_EXTDATAWIDTH = 0x1;
			break;
		default:
			// Dual data rate mode not still supported
			// HS_TIMING must be set to 0x1 before setting BUS_WIDTH for
			// dual data rate operation (values 5 or 6)
			return SDMMC_REQUEST_ERROR;
		}

		SDMMC.SDMASYSADDR.R = (uint32_t)&card_ext;
		result = sdmmc_send_cmd(SDMMC_CMD19_BUSTEST_W, 0x0,
								card_op.write_block_length,
								card_op.data_timeout);
		if (result != SDMMC_NO_ERROR)
			return result;
		sdmmc_wait_hd(HD_STATE_BTST, 0xff);
	}
	if (card_sm == HD_STATE_BTST) {
		memset(&card_ext, 0x0, card_op.write_block_length);
		PCM_0.PCM3.B.EMMC_MSTR_EN_BYTE_SWAP = 0x0;
		SDMMC.SDMASYSADDR.R = (uint32_t)&card_ext;
		result = sdmmc_send_cmd(SDMMC_CMD14_BUSTEST_R,
								0x0, card_op.read_block_length,
								card_op.data_timeout);
		PCM_0.PCM3.B.EMMC_MSTR_EN_BYTE_SWAP = 0x1;
		sdmmc_wait_hd(HD_STATE_TRAN, 0xff);
		if (result != SDMMC_NO_ERROR)
			return result;

		/* 31.1 For 8 data lines the mask is (MSB to LSB): 0x0000_0000_0000_FFFF
		 * 31.2 For 4 data lines the mask is (MSB to LSB): 0x0000_00FF
		 * 31.3-For 1 data line the mask is 0xC0
		 */
		if ((bus == SDMMC_BUS_1BIT_SINGLE) && (card_ext[0] != 0xc0))
			result = SDMMC_REQUEST_ERROR;
		if ((bus == SDMMC_BUS_4BIT_SINGLE) && (card_ext[0] != 0xff))
			result = SDMMC_REQUEST_ERROR;
		if ((bus == SDMMC_BUS_8BIT_SINGLE) && ((card_ext[0] != 0xff) || (card_ext[1] != 0xff)))
			result = SDMMC_REQUEST_ERROR;
	}

	/* We should check and change if necessary the power class depending of
	 * bus width request but we take the default for now. TBD.
	 */

	if (result == SDMMC_NO_ERROR) {
		// BUS_WIDTH [183]
		result = sdmmc_config_device_property(183, bus, 0x0);
	}
	if (result != SDMMC_NO_ERROR) {
		// 1BIT mode should work always otherwise we are into a fatal issue.
		SDMMC.HOSTCONTROL1.B.HOSTCTRL1_DATAWIDTH = 0x0;
		SDMMC.HOSTCONTROL1.B.HOSTCTRL1_EXTDATAWIDTH = 0x0;
		card_op.bus_width = SDMMC_BUS_1BIT_SINGLE;
		sdmmc_clear_error();
	} else
		card_op.bus_width = bus;
	return result;
}

/**
 * @brief				Retrieve the device identify information.
 *
 * @return				The data info, NULL otherwise.
 *
 * @api
 */
sdmmc_card_id_t *sdmmc_get_identification(void) {
	if (card_id.manufactur_id != 0x0)
		return &card_id;
	return NULL;
}

/**
 * @brief				Retrieve the device operation information.
 *
 * @return				The data info, NULL otherwise.
 *
 * @api
 */
sdmmc_card_op_t *sdmmc_get_operation(void) {
	// Return operation data if already discovered
	if (card_op.device_version != 0x0)
		return &card_op;
	return NULL;
}

/**
 * @brief				Initialize the host controller and device card.
 *
 * @return				No error if the request can be satisfied, sdmmc_result_t error otherwise.
 *
 * @api
 */
sdmmc_result_t sdmmc_init(void) {
	sdmmc_result_t result = SDMMC_CARD_ERROR;
	uint32_t reg32;
	uint8_t retries = 3;

	// Check if already identified
	if (card_id.manufactur_id != 0x0)
		return SDMMC_NO_ERROR;

	/* The Host driver writes the boot timeout value into
	 * boot timeout control register as per the eMMC4.3+ specification
	 */
	SDMMC.BOOTTIMEOUTCNT.R = 0xfff;
	// Normal Interrupts status enable
	SDMMC.NORMALINTRSTSENA.R = 0x7fff;
	// Normal interrupts signal enable
	SDMMC.NORMALINTRSIGENA.R = 0x7fff;
	// Error  Interrupts status enable
	SDMMC.ERRORINTRSTSENA.R = 0x137f;
	// Error  interrupts signal enable
	SDMMC.ERRORINTRSIGENA.R = 0x17ff;
	// Card reset values at boot time
	card_sm = HD_STATE_IDLE;

	/* The card is embedded than should be present in case
	 * of hardware reset propagation but not after software reset
	 * This will be improved when removable Card will be tested */
	result = sdmmc_wait_hc(HC_LEVEL_STATUS, HC_STATE_CARD, 0xfff);

	// Software reset
	sdmmc_reset(HC_STATE_DAT | HC_STATE_CMD);

	// Use default swap bytes configuration
	PCM_0.PCM3.B.EMMC_MSTR_EN_BYTE_SWAP = 0x1;
	PCM_0.PCM3.B.EMMC_SLV_EN_BYTE_SWAP = 0x1;

	// Program the core config
	reg32 = SDMMC.CORE_CONFIG.R & 0xffffff00UL;
	reg32|= SDMMC_ICLK / 1000 / 1000;
	SDMMC.CORE_CONFIG.R = reg32;

	// Program the feedback clock.
	SDMMC.FB_CLK_SEL.R = 0x3;

	// Setup SDMMC clock output for power-up sequence
	if (sdmmc_set_clock(HC_CLOCK_POWER) != SDMMC_NO_ERROR)
		return SDMMC_CLOCK_ERROR;

	// Timeout control
	SDMMC.TIMEOUTCONTROL.B.TIMEOUT_CTRVALUE = 0xf - 0x1;

	// The power can be enabled only after to setup voltage level.
	// Only 3.3V is supported */
	SDMMC.POWERCONTROL.B.PWRCTRL_SDBUSVOLTAGE = 0x7;
	SDMMC.POWERCONTROL.B.SD_BUS_POWER = 0x1;
	PMCDIG.VSIO.B.VSIO_EMMC = 0x0;

	// Power stabilization
	osalThreadDelayMilliseconds(5);

    // Card initialization
	do {
		result = sdmmc_card_initialization();
		if (result == SDMMC_NO_ERROR) {
			// Switching to high-speed mode
			result = sdmmc_switching_speed(SDMMC_SPEED_HIGH);
		}
		// Changing the data bus width to try to use max throughput
		if (result == SDMMC_NO_ERROR) {
			result = sdmmc_changing_data_bus(SDMMC_BUS_8BIT_SINGLE);
			if (result != SDMMC_NO_ERROR)
				result = sdmmc_changing_data_bus(SDMMC_BUS_4BIT_SINGLE);
				if (result != SDMMC_NO_ERROR)
					result = sdmmc_changing_data_bus(SDMMC_BUS_1BIT_SINGLE);
		}
	if (result == SDMMC_NO_ERROR)
		break;
	sdmmc_clear_error();
	} while (--retries != 0x0);

	return result;
}

sdmmc_result_t sdmmc_deinit(void) {
	memset(&card_id, 0x0, sizeof(card_id));
	memset(&card_op, 0x0, sizeof(card_op));
	// Clear any pending errors
	sdmmc_clear_error();
	// Reset lines for next Card discovering
	sdmmc_reset(HC_STATE_DAT | HC_STATE_CMD);
	// Power-off
	SDMMC.POWERCONTROL.B.SD_BUS_POWER = 0x0;
	// Normal Interrupts status disable
	SDMMC.NORMALINTRSTSENA.R = 0x0;
	// Normal interrupts signal disable
	SDMMC.NORMALINTRSIGENA.R = 0x0;
	// Error  Interrupts status disable
	SDMMC.ERRORINTRSTSENA.R = 0x0;
	// Error  interrupts signal disable
	SDMMC.ERRORINTRSIGENA.R = 0x0;
	SDMMC.BOOTTIMEOUTCNT.R = 0x0;
	return SDMMC_NO_ERROR;
}

/**
 * @brief				Write a writable unit.
 *						Write a single or a multiple writable unit. To retrieve the
 *						the supported write unit size can be used the sdmmc_get_write_size
 *						API.
 * @param[in] src		The source memory base address from wich to catch the data to write
 * 						of the single or multiple consecutive sectors.
 * @param[in] dst		The sector base address from wich to start the writing of the single
 * 						or multiple consecutive sectors.
 * @param[in] nbytes	Sector write size or multiple of one unit.
 *
 * @return				No error if the request can be satisfied, otherwise sdmmc_result_t error.
 *
 * @api
 */
sdmmc_result_t sdmmc_write(uint32_t *src, uint32_t *dst, uint32_t nbytes) {
	sdmmc_result_t result = SDMMC_SM_ERROR;
	uint32_t sdst = (uint32_t)dst;
	uint32_t ssrc = (uint32_t)src;

	// Data write
	if (card_id.manufactur_id == 0x0)
		return SDMMC_CARD_ERROR;

	// Nothing to do
	if (nbytes == 0x0)
		return SDMMC_NO_ERROR;

	sdmmc_switching_state(card_op.device_rca, HD_STATE_TRAN);

	if (card_sm == HD_STATE_TRAN) {
		result = sdmmc_send_cmd(SDMMC_CMD16_SET_BLOCKLEN,
								card_op.write_block_length, 0x0, 0xff);
		if (result != SDMMC_NO_ERROR)
			return result;

		if (nbytes < card_op.write_block_length) {
			result = sdmmc_read(dst, (uint32_t *)&card_ext,
								card_op.write_block_length);
			if (result != SDMMC_NO_ERROR)
				return result;
			// Write single-block
			memcpy(card_ext, src, nbytes);
			SDMMC.SDMASYSADDR.R = (uint32_t)card_ext;
			result = sdmmc_send_cmd(SDMMC_CMD24_WRITE_BLOCK,
									sdst, card_op.write_block_length, 0xff);
		}
		if (nbytes == card_op.write_block_length) {
			// Write single-block
			SDMMC.SDMASYSADDR.R = ssrc;
			result = sdmmc_send_cmd(SDMMC_CMD24_WRITE_BLOCK,
									sdst, card_op.write_block_length, 0xff);
		}
		if (nbytes > card_op.write_block_length) {
			uint32_t nsectors = nbytes / card_op.write_block_length;

			// Write multiple-block
			result = sdmmc_send_cmd(SDMMC_CMD23_SET_BLOCK_COUNT,
									nsectors, 0x0, 0xfff);
			if (result != SDMMC_NO_ERROR)
				return result;

			SDMMC.SDMASYSADDR.R = ssrc;
			result = sdmmc_send_cmd(SDMMC_CMD25_WRITE_MULTIPLE_BLOCK,
									sdst, nsectors * card_op.write_block_length,
									0xfff);
			if (result != SDMMC_NO_ERROR)
				return result;

			// Writing the remain bytes
			sdst += (nsectors * card_op.write_block_length);
			ssrc += (nsectors * card_op.write_block_length);
			result = sdmmc_write((uint32_t *)ssrc, (uint32_t *)sdst,
					nbytes - (nsectors * card_op.read_block_length));
		}
	} else result = SDMMC_SM_ERROR;

	return result;
}

/**
 * @brief				Read a readable unit.
 *						Read a single or a multiple readable unit. To retrieve the
 *						the supported read unit size can be used the sdmmc_get_read_size
 *						API.
 * @param[in] src		The sector base address from wich to start the reading of the single
 * 						or multiple consecutive sectors.
 * @param[in] dst		The destination memory base address from which to preserve the reading
 * 						of the single or multiple consecutive sectors.
 * @param[in] nbytes	Sector read size or multiple of one unit.
 *
 * @return				No error if the request can be satisfied, otherwise sdmmc_result_t error.
 *
 * @api
 */
sdmmc_result_t sdmmc_read(uint32_t *src, uint32_t *dst, uint32_t nbytes) {
	sdmmc_result_t result = SDMMC_SM_ERROR;
	uint32_t sdst = (uint32_t)dst;
	uint32_t ssrc = (uint32_t)src;

	// Data read
	if (card_id.manufactur_id == 0x0)
		return SDMMC_CARD_ERROR;

	// Nothing to do
	if (nbytes == 0x0)
		return SDMMC_NO_ERROR;

	sdmmc_switching_state(card_op.device_rca, HD_STATE_TRAN);

	if (card_sm == HD_STATE_TRAN) {
		result = sdmmc_send_cmd(SDMMC_CMD16_SET_BLOCKLEN,
								card_op.read_block_length, 0x0, 0xff);
		if (result != SDMMC_NO_ERROR)
			return result;
		if (nbytes <= card_op.read_block_length) {

			if (nbytes < card_op.read_block_length)
				sdst = (uint32_t)&card_ext;

			// Read single-block
			SDMMC.SDMASYSADDR.R = sdst;
			result = sdmmc_send_cmd(SDMMC_CMD17_READ_SINGLE_BLOCK,
									ssrc, card_op.read_block_length, 0xff);
			if (result != SDMMC_NO_ERROR)
				return result;

			if (nbytes < card_op.read_block_length)
				memcpy(dst, (uint8_t *)sdst, nbytes);
		} else {
			uint32_t nsectors = nbytes / card_op.read_block_length;

			// Read multiple-block
			result = sdmmc_send_cmd(SDMMC_CMD23_SET_BLOCK_COUNT,
									nsectors, 0x0, 0xff);
			if (result != SDMMC_NO_ERROR)
				return result;

			SDMMC.SDMASYSADDR.R = sdst;
			result = sdmmc_send_cmd(SDMMC_CMD18_READ_MULTIPLE_BLOCK,
									ssrc, nsectors * card_op.read_block_length,
									0xff);
			if (result != SDMMC_NO_ERROR)
				return result;

			// Reading the remaining bytes
			sdst += (nsectors * card_op.read_block_length);
			ssrc += (nsectors * card_op.read_block_length);
			result = sdmmc_read((uint32_t *)ssrc, (uint32_t *)sdst,
					nbytes - (nsectors * card_op.read_block_length));
		}
	} else
		result = SDMMC_SM_ERROR;

	return result;
}

/**
 * @brief				Erase a erasable unit.
 *						Erase a single or a multiple eresable unit. To retrieve the
 *						the supported erase unit size can be used the sdmmc_get_erase_size
 *						API.
 * @param[in] dst		The sector base address from wich to start the erasing of the single
 * 						or multiple consecutive sectors.
 * @param[in] nbytes	Sector erase size or multiple of one unit.
 *
 * @return				No error if the request can be satisfied, otherwise sdmmc_result_t error.
 *
 * @api
 */
sdmmc_result_t sdmmc_erase(uint32_t *dst, uint32_t nbytes) {
	sdmmc_result_t result = SDMMC_NO_ERROR;
	uint32_t bstart, bend;

	// Erase
	if (card_id.manufactur_id == 0x0)
		return SDMMC_CARD_ERROR;

	if (nbytes % card_op.erase_size)
		return SDMMC_REQUEST_ERROR;

	sdmmc_switching_state(card_op.device_rca, HD_STATE_TRAN);

	if (card_sm == HD_STATE_TRAN) {
		bstart = (uint32_t)dst;
		result = sdmmc_send_cmd(SDMMC_CMD35_ERASE_GROUP_STAR_T,
								bstart, 0x0, 0xff);
		if (result == SDMMC_NO_ERROR) {
			bend = (uint32_t)dst + nbytes;
			result = sdmmc_send_cmd(SDMMC_CMD36_ERASE_GROUP_END,
									bend, 0x0, 0xff);
			if (result == SDMMC_NO_ERROR) {
				result = sdmmc_send_cmd(SDMMC_CMD38_ERASE, 0x0, 0x0, 0xff);
			}
		}
	} else {
		result = SDMMC_SM_ERROR;
	}
	return result;
}

/**
 * @brief				Retrieve the basic readable unit.
 *
 * @return				The readable size unit, otherwise zero as error.
 *
 * @api
 */
uint16_t sdmmc_get_read_size(void) {
	if (card_id.manufactur_id != 0x0)
		return card_op.read_block_length;
	return 0x0;
}
/**
 * @brief				Retrieve the basic writable unit.
 *
 * @return				The writable size unit, otherwise zero as error.
 *
 * @api
 */
uint16_t sdmmc_get_write_size(void) {
	if (card_id.manufactur_id != 0x0)
		return card_op.write_block_length;
	return 0x0;
}

/**
 * @brief				Retrieve the basic erasable unit.
 *
 * @return				The erasable size unit, otherwise zero as error.
 *
 * @api
 */
uint16_t sdmmc_get_erase_size(void) {
	if (card_id.manufactur_id != 0x0)
		return card_op.erase_size;
	return 0x0;
}

/**
 * @brief				Switching power consumption or normal mode.
 *
 * @param[in] mode		The switch power mode request to move in power consumption or
 * 						normal operation mode.
 *
 * @return				No error if the request can be satisfied, otherwise sdmmc_result_t error.
 *
 * @api
 */
sdmmc_result_t sdmmc_switching_power(sdmmc_switch_t mode) {
	if (card_id.manufactur_id == 0x0)
		return SDMMC_CARD_ERROR;

	sdmmc_switching_state(card_op.device_rca,
			mode == SDMMC_SWITCH_SLEEP ? HD_STATE_SLP : HD_STATE_STBY);

	if ((mode == SDMMC_SWITCH_SLEEP) && (card_sm == HD_STATE_SLP))
		return SDMMC_NO_ERROR;
	if ((mode == SDMMC_SWITCH_AWAKE) && (card_sm == HD_STATE_STBY))
		return SDMMC_NO_ERROR;

	return SDMMC_REQUEST_ERROR;
}

#endif /* LLD_USE_SDMMC */
/** @} */
