/****************************************************************************
*
* Copyright © 2019 STMicroelectronics - All Rights Reserved
*
* This software is licensed under SLA0098 terms that can be found in the
* DM00779817_1_0.pdf file in the licenses directory of this software product.
* 
* THIS SOFTWARE IS DISTRIBUTED "AS IS," AND ALL WARRANTIES ARE DISCLAIMED,
* INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*
*****************************************************************************/
/**
 * @file    octalspi_lld.c
 * @brief   SPC5xx OctalSPI control unit.
 *
 * @addtogroup OctalSPI
 * @{
 */
#include "octalspi_lld.h"

#if (LLD_USE_OCTALSPI == TRUE) || defined(__DOXYGEN__)

/* Memory type defs */
#define OSPI_MTYP_MICRON			0x0U	/**<  Micron mode, regular SPI protocol. */
#define OSPI_MTYP_MACRONIX			0x1U	/**<  Macromix RAM mode, regular SPI protocol. */
#define OSPI_MTYP_MACRONIX_RAM		0x3U	/**<  Macromix RAM mode, regular SPI protocol with addressing map. */
#define	OSPI_MTYP_HYPERBUS_MEM		0x4U	/**<  HyperBus Memory mode, Cypress Spansion protocol. */
#define OSPI_MTYP_HYPERBUS_REG		0x5U	/**<  HyperBus Register mode, addressing register space. */
#define OSPI_MTYP_DEFAULT			0x6U	/**<  HyperBus default device type */

/* External Device configuration */
#define OSPI_DEV_TYPE(memtype)		((memtype & 0x0007) << 24) /**<  Memory type. */
#define OSPI_DEV_SIZE(bytes)		((bytes   & 0x001F) << 16) /**<  Device size. */
#define OSPI_DEV_CSHT(cycles)		((cycles  & 0x0007) <<  8) /**<  Number of Clock cycles to take ChipSelect high. */
#define OSPI_DEV_CKCSHT(cycles)		((cycles  & 0x0007) <<  4) /**<  Number of Clock cycles at end of transaction. */
#define OSPI_DEV_FRCK(free)			((free    & 0x0001) <<  1) /**<  Free running clock. 1 is free running. */
#define OSPI_DEV_CKMODE(ckmode)		((ckmode  & 0x0001) <<  0) /**<  Chip select release high or low. */
#define OSPI_DEV_WRAPSIZE(wsize)	((wsize   & 0x0007) << 16) /**<  Wrap size to which the memory is configured. */
#define OSPI_DEV_PRESCALER(pscaler)	((pscaler & 0x00FF) <<  0) /**<  Clock prescaler. */
#define OSPI_DEV_CSBOUND(csbound)	((csbound & 0x001F) << 16) /**<  Max boundary transaction, 2^csbound bytes. */
#define OSPI_DEV_MAXTRAN(maxtran)	((maxtran & 0x00FF) <<  0) /**<  Max bytes transaction request. */

/* OctalSPI HyperBus latency */
#define OSPI_HB_RWL(rwtime)		((rwtime  & 0x00FF) << 16) /**<  HyperBus R/W recovery time. */
#define OSPI_HB_ATL(atime)		((atime   & 0x00FF) <<  8) /**<  HyperBus access time. */
#define OSPI_HB_WZL(wzero)		((wzero   & 0x0001) <<  1) /**<  HyperBus no latency. */
#define OSPI_HB_LM(mode)		((mode    & 0x0001) <<  0) /**<  HyperBus latency mode. */

#define OSPI_SET(y, mask)		(y |=  (mask))
#define OSPI_CLR(y, mask)		(y &= ~(mask))

/*===========================================================================*/
/* Driver interrupt handlers.                                                */
/*===========================================================================*/
/* This general interrupt vector handles only the 5 interrupts below for each
 * OctalSPI interface but not the IT_OSPI interrupts as by reference manual.
 * IRQ OctalSPI_1 [187,191] and OctalSPI_2 [193,197]
 */
typedef enum {
	OSPI_NO_TO = 0x1U,	// Timeout
	OSPI_NO_SM,			// Status match
	OSPI_NO_FT,			// FIFO threshold
	OSPI_NO_TC,			// Transfer complete
	OSPI_NO_TE			// Transfer error
} ospi_irq_t;

#define OSPI_NO_IRQ(id, irq)	(186U + irq + ((id - 1U) * 6U))
#define OSPI_IRQ_PRIORITY		INTC_PSR_ENABLE(INTC_PSR_MAINCORE, 8)

typedef struct {
	uint8_t mem_mapped;
	uint8_t default_latency;
	uint8_t default_devtype;
	uint8_t default_nocsn;
	ospi_irq_t current_irq;
	ospi_modes_t current_mode;
} ospi_drv_t;

/**
 * OcotalSPI Driver Interface. Only one OctalSPI interface
 * is available on cut.1 and two avaliable on cut.2.
 */
#if (OSPI_INSTANCES == 0x1U)
#define OSPI_MEMMAP1	0x70000000UL
#define OSPI_MEMMAP2	0x70000000UL
#define OSPI_MEMREG1	(&OCTALSPI)
#define OSPI_MEMREG2	(&OCTALSPI)
#else
#define OSPI_MEMMAP1	0x70000000UL
#define OSPI_MEMMAP2	0x60000000UL
#define OSPI_MEMREG1	(&OCTALSPI_1)
#define OSPI_MEMREG2	(&OCTALSPI_2)
#endif
static volatile struct OCTALSPI_tag *octalspi_tags[1U + OSPI_INSTANCES];
static uint32_t	*octalspi_mems[1U + OSPI_INSTANCES];
static ospi_drv_t ospi_drv[1U + OSPI_INSTANCES];

/*******************************************************************************
* Local function prototypes
*******************************************************************************/
static inline ospi_status_t ospi_communication_config(uint8_t id, uint8_t isauto);
static inline void ospi_set_address_phase(uint8_t id, ospi_phases_t *phase);
static inline void ospi_set_data_phase(uint8_t id, ospi_phases_t *phase);
static inline void ospi_set_instruction_phase(uint8_t id, ospi_phases_t *phase);
static inline void ospi_set_alternate_phase(uint8_t id, ospi_phases_t *phase);
static inline void ospi_set_dummy_phase(uint8_t id, ospi_phases_t *phase);
static void ospi_irq_handler(uint8_t id, ospi_irq_t irq);

/*******************************************************************************
* Local functions
*******************************************************************************/
static void ospi_irq_handler(uint8_t id, ospi_irq_t irq) {
	ospi_drv[id].current_irq = irq - OSPI_NO_IRQ(id, 0x0U);
	switch (ospi_drv[id].current_irq) {
	case OSPI_NO_TO: octalspi_tags[id]->CR.B.TOIE = 0x0U; break;
	case OSPI_NO_SM: octalspi_tags[id]->CR.B.SMIE = 0x0U; break;
	case OSPI_NO_FT: octalspi_tags[id]->CR.B.FTIE = 0x0U; break;
	case OSPI_NO_TC: octalspi_tags[id]->CR.B.TCIE = 0x0U; break;
	case OSPI_NO_TE: octalspi_tags[id]->CR.B.TEIE = 0x0U; break;
	}
}

static inline void ospi_set_address_phase(uint8_t id, ospi_phases_t *phase) {
	if (phase == NULL) {
		/* Use default configuration */
		octalspi_tags[id]->CCR.B.ADSIZE = 0x3U; // address size 32-bit
		octalspi_tags[id]->CCR.B.ADDTR = 0x1U; // double transfer required enabled
		octalspi_tags[id]->CCR.B.ADMODE = 0x4U; // address with 8 lines
		octalspi_tags[id]->WCCR.B.ADSIZE = 0x3U; // address size 32-bit
		octalspi_tags[id]->WCCR.B.ADDTR = 0x1U;  // double transfer enabled
		octalspi_tags[id]->WCCR.B.ADMODE = 0x4U; // address with 8 lines
	} else {
		octalspi_tags[id]->CCR.B.ADSIZE = phase->size; // address size
		octalspi_tags[id]->CCR.B.ADDTR = phase->enable_dtr; // double transfer
		octalspi_tags[id]->CCR.B.ADMODE = phase->signal; // address lines
		octalspi_tags[id]->WCCR.B.ADSIZE = phase->size; // address size
		octalspi_tags[id]->WCCR.B.ADDTR = phase->enable_dtr;  // double transfer
		octalspi_tags[id]->WCCR.B.ADMODE = phase->signal; // address lines
	}
}

static inline void ospi_set_data_phase(uint8_t id, ospi_phases_t *phase) {
	octalspi_tags[id]->CCR.B.DQSE = 0x1U; // Data strobe is mandatory enabled
	octalspi_tags[id]->WCCR.B.DQSE = 0x1U; // Data strobe is mandatory enabled
	if (phase == NULL) {
		/* Use default configuration */
		octalspi_tags[id]->CCR.B.DDTR = 0x1U; // double transfer rate enabled
		octalspi_tags[id]->CCR.B.DMODE = 0x4U; // data mode lines
		octalspi_tags[id]->CCR.B.DQSE = 0x1U; // data strobe enabled
		octalspi_tags[id]->WCCR.B.DDTR = 0x1U; // double transfer rate enabled
		octalspi_tags[id]->WCCR.B.DMODE = 0x4U; // data on 8 lines
		octalspi_tags[id]->WCCR.B.DQSE = 0x1U; // data strobe enabled
	} else {
		octalspi_tags[id]->CCR.B.DDTR = phase->enable_dtr; // double transfer rate
		octalspi_tags[id]->CCR.B.DMODE = phase->signal; // data mode lines
		octalspi_tags[id]->WCCR.B.DQSE = phase->enable_strobe; // data strobe
		octalspi_tags[id]->WCCR.B.DDTR = phase->enable_dtr; // double transfer rate
		octalspi_tags[id]->WCCR.B.DMODE = phase->signal; // data mode lines
	}
}

static inline void ospi_set_instruction_phase(uint8_t id, ospi_phases_t *phase) {
	if (phase == NULL) {
		/* Use default configuration */
		octalspi_tags[id]->CCR.B.ISIZE = 0x0U; // instruction size
		octalspi_tags[id]->CCR.B.IDTR = 0x0U; // double transfer
		octalspi_tags[id]->CCR.B.IMODE = 0x0U; // instruction address lines
		octalspi_tags[id]->CCR.B.SIOO = 0x0U; // send instruction only once
		octalspi_tags[id]->WCCR.B.ISIZE = 0x0U; // instruction size
		octalspi_tags[id]->WCCR.B.IDTR = 0x0U; // double transfer
		octalspi_tags[id]->WCCR.B.IMODE = 0x0U; // instruction mode lines
		octalspi_tags[id]->WCCR.B.SIOO = 0x0U; // send instruction only once
	} else {
		octalspi_tags[id]->CCR.B.ISIZE = phase->size; // instruction size
		octalspi_tags[id]->CCR.B.IDTR = phase->enable_dtr; // double transfer rate
		octalspi_tags[id]->CCR.B.IMODE = phase->signal; // instruction address lines
		octalspi_tags[id]->CCR.B.SIOO = phase->only_once; // send instruction only once
		octalspi_tags[id]->WCCR.B.ISIZE = phase->size; // instruction size
		octalspi_tags[id]->WCCR.B.IDTR = phase->enable_dtr; // double transfer rate
		octalspi_tags[id]->WCCR.B.IMODE = phase->signal; // instruction mode lines
		octalspi_tags[id]->WCCR.B.SIOO = phase->only_once; // send instruction only once
	}
}

static inline void ospi_set_alternate_phase(uint8_t id, ospi_phases_t *phase) {
	if (phase == NULL) {
		/* Use default configuration */
		octalspi_tags[id]->CCR.B.ABSIZE = 0x0U; // address size 8-bit
		octalspi_tags[id]->CCR.B.ABDTR = 0x0U;  // double transfer disabled
		octalspi_tags[id]->CCR.B.ABMODE = 0x0U; // No alternate instruction
		octalspi_tags[id]->WCCR.B.ABSIZE = 0x0U; // address size 8-bit
		octalspi_tags[id]->WCCR.B.ABDTR = 0x0U;  // double transfer disabled
		octalspi_tags[id]->WCCR.B.ABMODE = 0x0U; // No alternate instruction
	} else {
		octalspi_tags[id]->CCR.B.ABSIZE = phase->size; // address size
		octalspi_tags[id]->CCR.B.ABDTR = phase->enable_dtr;  // double transfer rate
		octalspi_tags[id]->CCR.B.ABMODE = phase->signal; // address lines
		octalspi_tags[id]->WCCR.B.ABSIZE = phase->size; // address size
		octalspi_tags[id]->WCCR.B.ABDTR = phase->enable_dtr;  // double transfer rate
		octalspi_tags[id]->WCCR.B.ABMODE = phase->signal; // address lines
	}
}

static inline void ospi_set_dummy_phase(uint8_t id, ospi_phases_t *phase) {
	if (phase == NULL) {
		/* Use default configuration */
		octalspi_tags[id]->TCR.B.DCYC = 0x0U; // No read dummy cycles
		octalspi_tags[id]->TCR.B.DHQC = 0x1U; // Delay held quarter enabled
		octalspi_tags[id]->TCR.B.SSHIFT = 0x0U; // Sample shift disabled
		octalspi_tags[id]->WTCR.B.DCYC = 0x0U; // No write dummy cycles
	} else {
		octalspi_tags[id]->TCR.B.DCYC = phase->dummy_cycles; // Number of the dummy cycles
		octalspi_tags[id]->TCR.B.DHQC = phase->enable_delay; // Delay held quarter cycle
		octalspi_tags[id]->TCR.B.SSHIFT = phase->enable_shift; // Sample shift
		octalspi_tags[id]->WTCR.B.DCYC = phase->enable_delay; // Write number of the dummy cycles
	}
}

static uint8_t ospi_isvalid(uint8_t id) {
	if ((id < OSPI_ID1) || (id > OSPI_ID2)) {
		return 0x0U;
	}
	if ((octalspi_tags[id] == 0x0U) ||
		(octalspi_mems[id] == 0x0U)) {
		return 0x0U;
	}

	return 0x1U;
}

static ospi_status_t ospi_wait_and_clear(uint8_t id) {
	ospi_status_t retval;

	/* As soon as a command is started, the BUSY bit (OCTOSPI_SR[5])
	 * is automatically set and this is could be the case of some memcpy/memset
	 * has been performed on main application.
	 */
	while ((retval = octalspi_get_status(id)) == OSPI_OP_BUSY) {
		/* Release BUSY bit to permit other driver access */
		octalspi_tags[id]->CR.B.EN = 0x0U;
		octalspi_tags[id]->CR.B.ABORT = 0x1U;
		octalspi_tags[id]->FCR.R = 0x1BU;
		octalspi_tags[id]->CR.B.EN = 0x1U;
	}
	return retval;
}

static ospi_status_t ospi_wait_for(uint8_t id, ospi_irq_t no_irq, ospi_status_t status) {
	ospi_status_t retval = OSPI_OP_COMPLETE;
	uint32_t ctime;

	/* Check if it is a transfer error notification */
	if (no_irq == OSPI_NO_TE) {
		retval = OSPI_OP_ERROR;
	}
	/* Wait the expected interrupt notification */
	if (retval == OSPI_OP_COMPLETE) {
		ctime = osalThreadGetMicroseconds();
		while (ospi_drv[id].current_irq != no_irq) {
			if (1000U < (osalThreadGetMicroseconds() - ctime)) {
				retval = OSPI_OP_TIMEOUT;
				break;
			}
		}
	}
	/* Check on status flag. */
	if (retval != OSPI_OP_COMPLETE) {
		ctime = osalThreadGetMicroseconds();
		while (octalspi_get_status(id) != status) {
			if (1000U < (osalThreadGetMicroseconds() - ctime)) {
				retval = octalspi_get_status(id);
				ospi_wait_and_clear(id);
				break;
			}
		}
	}
	/* Re-enable interrupt */
	switch (ospi_drv[id].current_irq) {
	case OSPI_NO_TO: /* Timeout managed by lowpower routine */
		if (octalspi_tags[id]->CR.B.TCEN == 0x1U) {
			octalspi_tags[id]->CR.B.TOIE = 0x1U;
		}
		octalspi_tags[id]->FCR.B.CTOF = 0x1U;
	break;
	case OSPI_NO_SM: /* Status match */
		octalspi_tags[id]->CR.B.SMIE = 0x1U;
		octalspi_tags[id]->FCR.B.CSMF = 0x1U;
	break;
	case OSPI_NO_FT: /* FIFO threshold */
		octalspi_tags[id]->CR.B.FTIE = 0x1U;
	break;
	case OSPI_NO_TC: /* Transfer complete */
		octalspi_tags[id]->CR.B.TCIE = 0x1U;
		octalspi_tags[id]->FCR.B.CTCF = 0x1U;
	break;
	case OSPI_NO_TE: /* Transfer error */
		octalspi_tags[id]->CR.B.TEIE = 0x1U;
		octalspi_tags[id]->FCR.B.CTEF = 0x1U;
	break;
	}
	return retval;
}
		
static ospi_status_t ospi_functional_config(uint8_t id, ospi_modes_t fmode, uint8_t devtype) {
	ospi_status_t retval;

	retval = ospi_wait_and_clear(id);
	/* The below state, complete or FIFO threshold, means ready to start
	 * a new transaction for the new switch functional context
	 */
	if ((retval == OSPI_OP_COMPLETE) || (retval == OSPI_OP_FIFOTHRESH)) {
		if (fmode == OSPI_INDIRECT_MODE) {
			octalspi_tags[id]->CR.B.FMODE = OSPI_INDIRECT_MODE & 0x3U;
			octalspi_tags[id]->CR.B.FTIE = 0x1U;
			octalspi_tags[id]->CR.B.TEIE = 0x1U;
			octalspi_tags[id]->CR.B.TCIE = 0x1U;
		} else if (fmode == OSPI_POLLING_MODE) {
			octalspi_tags[id]->CR.B.FMODE = OSPI_POLLING_MODE & 0x3U;
			octalspi_tags[id]->CR.B.SMIE = 0x1U;
			octalspi_tags[id]->CR.B.FTIE = 0x0U;
		} else {
			octalspi_tags[id]->CR.B.FMODE = OSPI_MEM_MAPPED_MODE & 0x3U;
		}

		if (devtype != OSPI_MTYP_DEFAULT) {
			octalspi_tags[id]->DCR1.B.MTYP = devtype;
		}
		ospi_drv[id].current_mode = fmode;
	}
	return retval;
}

static void ospi_device_config(uint8_t id) {
	uint32_t regval, devsize = 0x0U;
	uint32_t maxsize = OSPI_FLASHSIZE;

	if (id == OSPI_ID2) {
		maxsize = OSPI_RAMSIZE;
	}
	
	/* 2^(devsize-1) is the total device size addressable */
	while (maxsize >>= 0x1U) devsize++;
	devsize -= 0x1U;
	
	ospi_wait_and_clear(id);

	/* Set default device type */
	ospi_drv[id].default_devtype = OSPI_MEMTYPE;

	/* Device configuration registers */
	regval = 0x0U;
	OSPI_SET(regval, OSPI_DEV_TYPE(OSPI_MEMTYPE));
	OSPI_SET(regval, OSPI_DEV_SIZE(devsize));
	OSPI_SET(regval, OSPI_DEV_CSHT(0x0U));
	OSPI_SET(regval, OSPI_DEV_CKCSHT(0x0U));
	OSPI_SET(regval, OSPI_DEV_FRCK(0x0U));
	OSPI_SET(regval, OSPI_DEV_CKMODE(0x0U));
	octalspi_tags[id]->DCR1.R = regval;

	regval = 0x0U;
	OSPI_SET(regval, OSPI_DEV_WRAPSIZE(0x0U));
	OSPI_SET(regval, OSPI_DEV_PRESCALER(0x1U)); // x means FCLOCK/(x+1)
	octalspi_tags[id]->DCR2.R =  regval;

	regval = 0x0U;
	OSPI_SET(regval, OSPI_DEV_CSBOUND(0x0U));
	OSPI_SET(regval, OSPI_DEV_MAXTRAN(0x0U));
	octalspi_tags[id]->DCR3.R = regval;
}

static ospi_status_t ospi_ip_config(uint8_t id) {
	ospi_status_t retval = OSPI_OP_REQUEST;

	retval = ospi_wait_and_clear(id);
	if (retval != OSPI_OP_COMPLETE)
		return retval;

	/* OctalSPI IP configuration */
	octalspi_tags[id]->CR.B.FTHRES = 0x0U;
	octalspi_tags[id]->CR.B.DQM = 0x0U;

	/* Low-power mode disabled at start-up. */
	octalspi_tags[id]->LPTR.R = 0x0U;
	octalspi_tags[id]->CR.B.TCEN = 0x0U;

	/* Set default functional mode operation */
	ospi_drv[id].current_mode = OSPI_MEM_MAPPED_MODE;

	return ospi_functional_config(id, ospi_drv[id].current_mode, OSPI_MTYP_DEFAULT);
}

static ospi_status_t ospi_communication_config(uint8_t id, uint8_t isauto){
	ospi_status_t retval = OSPI_OP_REQUEST;

	if (isauto == 0x1U) {
		return OSPI_OP_COMPLETE;
	}

	retval = ospi_wait_and_clear(id);
	if (retval != OSPI_OP_COMPLETE)
		return retval;

	/* Instruction phase configuration */
	ospi_set_instruction_phase(id, NULL);

	/* Address phase configuration */
	ospi_set_address_phase(id, NULL);

	/* Alternate bytes mode phase */
	ospi_set_alternate_phase(id, NULL);

	/* Data phase configuration */
	ospi_set_data_phase(id, NULL);

	/* Dummy cycles phase */
	ospi_set_dummy_phase(id, NULL);

	return OSPI_OP_COMPLETE;
}

static ospi_status_t ospi_init(uint8_t id) {
	ospi_status_t retval = OSPI_OP_REQUEST;
	ospi_hwinfo_t hwinfo;
	uint8_t i;
	
	/* Check if already initialized */
	if (octalspi_tags[id] != 0x0U)
		return retval;
	/* Check for valid input clock */
	if (OSPI_ICLK < 200000000U)
		return retval;

	/* Enable peripherical only one time */
	if (octalspi_mems[0x0U] == 0x0U) {
		SPCSetPeripheralClockMode(OSPI_PCTL, 
			SPC5_ME_PCTL_RUN(0x1U) | SPC5_ME_PCTL_LP(0x2U));
		/* Enable OctalSPI 1 interrupts */
		for (i = OSPI_NO_TO; i <= OSPI_NO_TE; i++) {
			INTC_PSR(OSPI_NO_IRQ(OSPI_ID1, i)) = OSPI_IRQ_PRIORITY;
		}
		/* There is only one register into OctalSPI IO manager 
		 * at address 0x80002000, which should be 0x00000001 
		 * for dual-CSN operation.
		 */
		if (OSPI_INSTANCES == 0x2U) {
			*(vuint32_t *)((uint32_t)OSPI_MEMREG1 + 0x2000U) = 0x1U;
			/* Enable OctalSPI 2 interrupts */
			for (i = OSPI_NO_TO; i <= OSPI_NO_TE; i++) {
				INTC_PSR(OSPI_NO_IRQ(OSPI_ID2, i)) = OSPI_IRQ_PRIORITY;
			}
		}
		octalspi_mems[0x0U] += 0x1U;
	}

	/* Wait for clock normalization */
	osalThreadDelayMilliseconds(0x10);

	/* OctalSPI reference tag initialization */
	if (id == OSPI_ID1) {
		octalspi_tags[OSPI_ID1] = OSPI_MEMREG1;
		octalspi_mems[OSPI_ID1] = (uint32_t *)OSPI_MEMMAP1;
	} else {
		octalspi_tags[OSPI_ID2] = OSPI_MEMREG2;
		octalspi_mems[OSPI_ID2] = (uint32_t *)OSPI_MEMMAP2;
	}

	/* Device configuration */
	ospi_device_config(id);

	/* Define communication type */
	ospi_communication_config(id, 0x0U);

	/* IP configuration */
	ospi_ip_config(id);

	/* Clear all status flags */
	octalspi_tags[id]->FCR.R = 0x1BU;

	/* Hyperbus latency configuration */
	retval = octalspi_set_hyperbus(id, NULL);

	/* Enable OctalSPI */
	octalspi_tags[id]->CR.B.EN = 0x1U;

	/* Check if memory-mapped write is supported */
	if (octalspi_get_hwinfo(id, &hwinfo) == OSPI_OP_COMPLETE) {
		if (hwinfo.ospi_mem_mapped == 0x1U)
			ospi_drv[id].mem_mapped = 0x1U;
	}
	return retval;
}
/*******************************************************************************
* Global functions
*******************************************************************************/
/**
 * @brief	Initialize OctalSPI interface to be usable.
 *
 * @return	No error if OctalSPI is initialized, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_init(void) {
	ospi_status_t retval = OSPI_OP_REQUEST;
	
	/* Initialize OctalSPI 1 interface */
	retval = ospi_init(OSPI_ID1);
	ospi_drv[0x0U].default_nocsn = OSPI_ID1;
	if ((retval == OSPI_OP_COMPLETE) && (OSPI_INSTANCES == 0x2U)) {
		/* Initialize OctalSPI 2 interface */
		retval = ospi_init(OSPI_ID2);
		ospi_drv[0x0U].default_nocsn = OSPI_ID2;
	}
	return retval;
}

/**
 * @brief				OctalSPI get base address for each memory instances.
 *
 * @param[in] id		OSPI_ID1 for HyperRAM and OSPI_ID2 for HyperFLASH
 * 						base addresses request
 *
 * @return				No error if OctalSPI request can be satisfied, NULL address otherwise.
 *
 * @api
 */
uint32_t octalspi_get_baseaddr(uint8_t id) {
	if (ospi_isvalid(id) == 0x1U) {
		return (uint32_t)octalspi_mems[id];
	}
	return 0x0U;
}

/**
 * @brief   			OctalSPI write operation.
 *
 * @param[in] src		The source non OctalSPI address.
 * @param[in] dst		The destination OctalSPI address. The base address can be discover via
 * 						octalspi_get_baseaddr routine using OSPI_ID1 id to know the HyperRAM base address
 * 						and OSPI_ID2 for HyperFLASH base address.
 * @param[in] nbytes	The number of the byte to read.
 *
 * @return				No error if OctalSPI request can be satisfied, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_write(uint32_t *src, uint32_t *dst, uint32_t nbytes) {
	ospi_status_t retval = OSPI_OP_REQUEST;
	uint8_t id,reqtype;
	uint32_t baddr, bfr[0x1U], w;
	uint8_t b,*bsrc, *bbfr;

	if (nbytes == 0x0U)
		return OSPI_OP_COMPLETE;

	/* Data source must be outside of Hyperbus memory space because not
	 * still supported */
	if (((uint32_t)src >= OSPI_MEMMAP1) &&
		(((uint32_t)src + nbytes)) < (OSPI_MEMMAP2 + OSPI_FLASHSIZE)) {
		return retval;
	}

	/* Suppose a HyperFlash request command */
	id = OSPI_ID2;
	reqtype = OSPI_MTYP_HYPERBUS_REG;

	/* Check if a HyperRAM request */
	if (((uint32_t)dst >= OSPI_MEMMAP1) &&
		(((uint32_t)dst + nbytes)) < (OSPI_MEMMAP1 + OSPI_RAMSIZE )) {
		id = OSPI_ID1;
		reqtype = OSPI_MTYP_DEFAULT;
	} else if (((uint32_t)dst >= OSPI_MEMMAP2) &&
			(((uint32_t)dst + nbytes)) < (OSPI_MEMMAP2 + OSPI_FLASHSIZE)) {
			id = OSPI_ID1;
			reqtype = OSPI_MTYP_DEFAULT;
	}

	/* Check if available only one OctalSPI interface */
	if (ospi_drv[0x0U].default_nocsn == 0x1U)
		id = OSPI_ID1;

	/* Clear in case of FIFO full due same external debug request
	 * The driver must start transaction from a clear scenarious
	 */
	ospi_wait_and_clear(id);

	if (reqtype == OSPI_MTYP_DEFAULT) {
		/*
		 * Received HyperRAM write request
		 */
		baddr = (uint32_t)dst & (~OSPI_MEMMAP1);

		switch (ospi_drv[id].current_mode) {
		case OSPI_INDIRECT_MODE:
			octalspi_tags[id]->CR.B.FMODE = (OSPI_INDIRECT_MODE & 0x3U) - 0x1U;
			octalspi_tags[id]->CR.B.FTHRES = sizeof(uint32_t);
			/* Writing data words */
			if (nbytes <= sizeof(uint32_t)) {
				w = 0x0U;
				octalspi_tags[id]->DLR.R = sizeof(uint32_t) - 0x1U;
				octalspi_tags[id]->AR.R = baddr;
			} else {
				w = nbytes / sizeof(uint32_t);
				w *= sizeof(uint32_t);
				octalspi_tags[id]->DLR.R = w - 0x1U;
				octalspi_tags[id]->AR.R = baddr;

				/* Wait for FIFO freed */
				retval = ospi_wait_for(id, OSPI_NO_FT, OSPI_OP_FIFOTHRESH);

				/* Writing data */
				for (w = 0; w < (nbytes / sizeof(uint32_t)); w++) {
					*((uint32_t *) &octalspi_tags[id]->DR.R) = src[w];
					baddr += sizeof(uint32_t);
					octalspi_tags[id]->AR.R = baddr;
				}

				/* Wait for transfer complete */
				retval = ospi_wait_for(id, OSPI_NO_TC, OSPI_OP_COMPLETE);
			}

			/* Writing latest odd data */
			w *= sizeof(uint32_t);
			if (nbytes != w) {
				bsrc = (uint8_t *)src+w;
				bbfr = (uint8_t *)bfr;
				octalspi_read((uint32_t *)(baddr|OSPI_MEMMAP1), bfr, sizeof(uint32_t));
				for (b=0; b < (nbytes - w); b++) {
					bbfr[b] = bsrc[b];
				}
				octalspi_tags[id]->CR.B.FMODE = (OSPI_INDIRECT_MODE & 0x3U) - 0x1U;
				octalspi_tags[id]->DLR.R = 0x3U;
				octalspi_tags[id]->AR.R = baddr;
				*((uint32_t *) &octalspi_tags[id]->DR.R) = bfr[0U];
			}

			/* Wait for transfer complete */
			retval = ospi_wait_for(id, OSPI_NO_TC, OSPI_OP_COMPLETE);

			/* FIFO must became free */
			if ((retval = octalspi_get_status(id)) == OSPI_OP_FIFOTHRESH)
				retval = OSPI_OP_COMPLETE;
			break;
		case OSPI_POLLING_MODE:
			/* The status polling mode can be only a read */
			retval = OSPI_OP_REQUEST;
			break;
		case OSPI_MEM_MAPPED_MODE:
			{
				if (!ospi_drv[id].mem_mapped) {
					return OSPI_OP_REQUEST;
				}

				for (w = 0; w < (nbytes / sizeof(uint32_t)); w++) {
					dst[w] = src[w];
				}
				/* Writing latest odd data */
				w = w * sizeof(uint32_t);
				if (nbytes != w) {
					bfr[0U] = dst[w];
					bsrc = (uint8_t *)src+w;
					bbfr = (uint8_t *)bfr;
					for (b=0; b < (nbytes - w); b++) {
						bbfr[b] = bsrc[b];
					}
					dst[w] = bfr[w];
				}
				/* Release BUSY bit to permit other external mode access */
				octalspi_tags[id]->CR.B.ABORT = 0x1U;
				break;
			}
		}
	} else {
		/*
		 * Received HyperFLASH write request
		 */
		ospi_modes_t cmode = OSPI_MEM_MAPPED_MODE;

		if (nbytes != sizeof(uint16_t)) {
			return OSPI_OP_REQUEST;
		}

		/* Save current context */
		octalspi_get_fmode(id, &cmode);
		baddr = (uint32_t)dst & (~OSPI_MEMMAP2);

		if (ospi_drv[id].default_latency == 0x1U)
			octalspi_tags[id]->HLCR.B.WZL = 0x1U;

		ospi_functional_config(id, OSPI_INDIRECT_MODE, OSPI_MTYP_HYPERBUS_REG);
		octalspi_tags[id]->CR.B.FMODE = (OSPI_INDIRECT_MODE & 0x3U) - 0x1U;
		octalspi_tags[id]->CR.B.FTHRES = sizeof(uint16_t);

		/* Wait for FIFO freed */
		retval = ospi_wait_for(id, OSPI_NO_FT, OSPI_OP_FIFOTHRESH);

		/* Writing data */
		octalspi_tags[id]->DLR.R = nbytes-0x1U;
		octalspi_tags[id]->AR.R = baddr;
		*((uint16_t *) &octalspi_tags[id]->DR.R) = *(uint16_t *)src;

		/* Wait for transfer complete */
		retval |= ospi_wait_for(id, OSPI_NO_TC, OSPI_OP_COMPLETE);

		/* Restore the context.  */
		ospi_functional_config(id, cmode, ospi_drv[id].default_devtype);
		if (ospi_drv[id].default_latency == 0x1U)
			octalspi_tags[id]->HLCR.B.WZL = 0x0U;
	}
	/* Clear all status flags */
	octalspi_tags[id]->FCR.R = 0x1BU;
	return retval;
}

/**
 * @brief   			OctalSPI read operation.
 *
 * @param[in] src		The source non OctalSPI address.
 * @param[in] dst		The destination OctalSPI address. The base address can be discover via
 * 						octalspi_get_baseaddr routine using OSPI_ID2 id to know the HyperRAM base address
 * 						and OSPI_ID2 for HyperFLASH base address.
 * @param[in] nbytes	The number of the byte to read.
 *
 * @return				No error if OctalSPI request can be satisfied, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_read(uint32_t *src, uint32_t *dst, uint32_t nbytes) {
	uint8_t id,reqtype;
	ospi_status_t retval = OSPI_OP_REQUEST;
	uint32_t w, baddr, bfr[1U];
	uint8_t b,*bdst, *bbfr;

	if (nbytes == 0U)
		return OSPI_OP_COMPLETE;

	/* Data must be inside of Hyperbus memory space */
	if (((uint32_t)dst >= OSPI_MEMMAP1) &&
		(((uint32_t)dst + nbytes)) <= (OSPI_MEMMAP2 + OSPI_FLASHSIZE)) {
		return retval;
	}

	/* Suppose a HyperFlash request command */
	id = OSPI_ID2;
	reqtype = OSPI_MTYP_HYPERBUS_REG;

	/* Check if a HyperRAM request */
	if (((uint32_t)src >= OSPI_MEMMAP1) &&
		(((uint32_t)src + nbytes)) < (OSPI_MEMMAP1 + OSPI_RAMSIZE)) {
		id = OSPI_ID1;
		reqtype = OSPI_MTYP_DEFAULT;
	} else if (((uint32_t)src >= OSPI_MEMMAP2) &&
			(((uint32_t)src + nbytes)) < (OSPI_MEMMAP2 + OSPI_FLASHSIZE)) {
			id = OSPI_ID1;
			reqtype = OSPI_MTYP_DEFAULT;
	}

	/* Check if available only one OctalSPI interface */
	if (ospi_drv[0x0U].default_nocsn == 0x1U)
		id = OSPI_ID1;

	/* Clear in case of FIFO full due same external debug request
	 * The driver must start transaction from a clear scenarious */
	ospi_wait_and_clear(id);

	if (reqtype == OSPI_MTYP_DEFAULT) {
		baddr = (uint32_t)src & (~OSPI_MEMMAP1);

		switch (ospi_drv[id].current_mode) {
		case OSPI_INDIRECT_MODE:
			octalspi_tags[id]->CR.B.FTHRES = sizeof(uint32_t);
			octalspi_tags[id]->CR.B.FMODE = (OSPI_INDIRECT_MODE & 0x3U);
			if (nbytes <= sizeof(uint32_t)) {
				/* Read only one word */
				w = 0x0U;
				octalspi_tags[id]->DLR.R = sizeof(uint32_t) - 0x1U;
				octalspi_tags[id]->AR.R = baddr;
			} else {
				/* Read one word plus */
				w = (nbytes / sizeof(uint32_t)) + 0x1U;
				w *= sizeof(uint32_t);
				octalspi_tags[id]->DLR.R = w - 0x1U;
				octalspi_tags[id]->AR.R = baddr;
				/* In indirect-read mode, the FTF is set when the number of
				 * valid bytes to be read from the FIFO is above the threshold
				 * Wait for FIFO complete.
				 */
				retval = ospi_wait_for(id, OSPI_NO_FT, OSPI_OP_FIFOTHRESH);

				/* Reading data */
				for (w=0; w < nbytes / sizeof(uint32_t); w++) {
					dst[w] = *((uint32_t *) &octalspi_tags[id]->DR.R);
					baddr += sizeof(uint32_t);
				}
			}

			/* Reading latest odd data */
			w *= sizeof(uint32_t);
			if ((nbytes < sizeof(uint32_t)) || (nbytes != w)) {
				bfr[0U] = *((uint32_t *) &octalspi_tags[id]->DR.R);
			}
			if (nbytes != w) {
				uint8_t *bdst = (uint8_t *)dst+w;
				uint8_t *bbfr = (uint8_t *)bfr;
				for (b=0; b < (nbytes - w); b++) {
					bdst[b] = bbfr[b];
				}
			}

			/* Wait for transfer complete */
			retval = ospi_wait_for(id, OSPI_NO_TC, OSPI_OP_COMPLETE);

			/* FIFO must became free */
			if ((retval = octalspi_get_status(id)) == OSPI_OP_FIFOTHRESH)
				retval = OSPI_OP_COMPLETE;
			break;
		case OSPI_POLLING_MODE:
			octalspi_tags[id]->CR.B.FTHRES = sizeof(uint32_t);
			octalspi_tags[id]->CR.B.FMODE = (OSPI_POLLING_MODE & 0x3U);
			octalspi_tags[id]->DLR.R = nbytes - 0x1U;
			octalspi_tags[id]->AR.R = baddr;

			/* Wait for status match complete */
			retval = ospi_wait_for(id, OSPI_NO_SM, OSPI_OP_MATCHDATA);

			/* Reading the matched data word */
			if (retval == OSPI_OP_COMPLETE) {
				dst[0x0U] = *((uint32_t *) &octalspi_tags[id]->DR.R);
			}
			break;
		case OSPI_MEM_MAPPED_MODE:
			for (w = 0; w < (nbytes / sizeof(uint32_t)); w++) {
				dst[w] = src[w];
			}
			/* Reading latest odd data */
			w = w * sizeof(uint32_t);
			if (nbytes != w) {
				bfr[0U] = src[w];
				bdst = (uint8_t *)dst+w;
				bbfr = (uint8_t *)bfr;
				for (b=0; b < (nbytes - w); b++) {
					bdst[b] = bbfr[b];
				}
			}
			/* Release BUSY bit to permit other external mode access */
			octalspi_tags[id]->CR.B.ABORT = 0x1U;
			break;
		}
	} else {
		/*
		 * Received HyperFLASH read request
		 */
		ospi_modes_t cmode = OSPI_MEM_MAPPED_MODE;

		if (nbytes != sizeof(uint16_t)) {
			return OSPI_OP_REQUEST;
		}

		/* Save current context */
		octalspi_get_fmode(id, &cmode);
		baddr = (uint32_t)src & (~OSPI_MEMMAP2);

		/* Send the command */
		ospi_functional_config(id, OSPI_INDIRECT_MODE, OSPI_MTYP_HYPERBUS_REG);
		octalspi_tags[id]->CR.B.FMODE = (OSPI_INDIRECT_MODE & 0x3U);
		octalspi_tags[id]->CR.B.FTHRES = sizeof(uint16_t);
		octalspi_tags[id]->DLR.R = nbytes-0x1U;
		octalspi_tags[id]->AR.R = baddr;

		/* Wait for transfer complete. Data ready in FIFO */
		retval = ospi_wait_for(id, OSPI_NO_TC, OSPI_OP_COMPLETE);

		/* Reading data */
		*(uint16_t *)dst = (uint16_t)octalspi_tags[id]->DR.R;

		/* Restore the current functional mode */
		ospi_functional_config(id, cmode, ospi_drv[id].default_devtype);
		octalspi_tags[id]->HLCR.B.WZL = 0x1U;
	}
	/* Clear all status flags */
	octalspi_tags[id]->FCR.R = 0x1BU;
	return retval;
}

/**
 * @brief	DeInitialize OctalSPI interface.
 *
 * @api
 */
void octalspi_deinit(void) {
	if (octalspi_mems[0] != 0U) {
		SPCSetPeripheralClockMode(OSPI_PCTL,
			SPC5_ME_PCTL_RUN(0) | SPC5_ME_PCTL_LP(0));
	}

	octalspi_mems[0] = 0U;
	octalspi_tags[OSPI_ID1] = octalspi_tags[OSPI_ID2] = 0U;
	octalspi_mems[OSPI_ID1] = octalspi_mems[OSPI_ID2] = 0U;
	return;
}

/**
 * @brief   			Get the current OctalSPI status.
 *
 * @param[in] id		OSPI_ID1 to dial with OctalSPI 1 interface, OSPI_ID2 otherwise for second interface.
 *
 * @return				The status.
 *
 * @api
 */
ospi_status_t octalspi_get_status(uint8_t id) {
	uint32_t regval;
	
	if (ospi_isvalid(id) != 1U) {
		return OSPI_OP_ERROR;
	}

	regval = octalspi_tags[id]->SR.R;

	/* Status bit depends of the current functional mode */
	switch (ospi_drv[id].current_mode) {
	case OSPI_INDIRECT_MODE:
		if ((regval & 0x4U) != 0x0U)	{ return OSPI_OP_FIFOTHRESH; }
		if ((regval & 0x2U) != 0x0U)	{ return OSPI_OP_COMPLETE; }
		if ((regval >> 0x8U) == 0x20U)	{ return OSPI_OP_FIFOFULL; }
		if ((regval & 0x1U) != 0x0U)	{ return OSPI_OP_ERROR; }
		break;
	case OSPI_POLLING_MODE:
		if ((regval & 0x8U) != 0x0U)	{ return OSPI_OP_MATCHDATA; }
		if ((regval & 0x4U) != 0x0U)	{ return OSPI_OP_FIFOTHRESH; }
		if ((regval & 0x2U) != 0x0U)	{ return OSPI_OP_ABORT; }
		break;
	case OSPI_MEM_MAPPED_MODE:
		if ((regval & 0x02U) != 0x0U) { return OSPI_OP_ABORT; }
		break;
	}
	/* Common Status Error for all functional modes */
	if ((regval & 0x10U) != 0x0U) {	return OSPI_OP_TIMEOUT;	}
	if ((regval & 0x20U) != 0x0U) {	return OSPI_OP_BUSY; }

	return OSPI_OP_COMPLETE;
}

/**
 * @brief   			Set a functional mode.
 *
 * @param[in] id		OSPI_ID1 to dial with OctalSPI 1 interface, OSPI_ID2 otherwise for second interface.
 *
 * @param[in] fmode		The current functional mode to be configured in OctalsPI interface.
 *
 * @return				No error if OctalSPI request can be satisfied, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_set_fmode(uint8_t id, ospi_modes_t fmode) {
	ospi_status_t retval = OSPI_OP_REQUEST;
	
	if ((ospi_isvalid(id) != 1U) ||
		(fmode < OSPI_INDIRECT_MODE) ||
		(fmode > OSPI_MEM_MAPPED_MODE)) {
		return retval;
	}

	retval = ospi_functional_config(id, fmode, OSPI_MTYP_DEFAULT);

	/* Clear all status flags */
	octalspi_tags[id]->FCR.R = 0x1BU;
	return retval;
}

/**
 * @brief   			Get the current functional mode.
 *
 * @param[in] id		OSPI_ID1 to dial with OctalSPI 1 interface, OSPI_ID2 otherwise for second interface.
 *
 * @param[out] fmode	The current functional mode configured in OctalsPI interface.
 *
 * @return				No error if OctalSPI was initialized, OSPI_OP_REQUEST error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_get_fmode(uint8_t id, ospi_modes_t *fmode) {
	if (ospi_isvalid(id) != 0x1U) {
		return OSPI_OP_REQUEST;
	}
	*fmode = ospi_drv[id].current_mode;
	return OSPI_OP_COMPLETE;
}

/**
 * @brief   			Get OctalSPI hardware configuration info.
 *
 * @param[in] id		OSPI_ID1 to dial with OctalSPI 1 interface, OSPI_ID2 otherwise for second interface.
 *
 * @param[out] hwinfo	The hardware info discovered.
 *
 * @return				No error if OctalSPI was initialized, OSPI_OP_REQUEST error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_get_hwinfo(uint8_t id, ospi_hwinfo_t *hwinfo) {
	ospi_status_t retval = OSPI_OP_REQUEST;
	uint32_t regval;

	if (ospi_isvalid(id) != 0x1U) {
		return retval;
	}
	/* OctalSPI identification ID */
	regval = octalspi_tags[id]->ID.R;
	hwinfo->ospi_identification = regval;
	regval = octalspi_tags[id]->MID.R;
	hwinfo->ospi_magic_id = regval;

	/* OctalSPI version */
	regval = octalspi_tags[id]->VER.R;
	hwinfo->ospi_version = regval & 0x00FFU;
		
	/* OctalSPI HW configuration */
	regval = octalspi_tags[id]->HWCFGR.R;
	hwinfo->ospi_axi_interface = regval & 0x000FU;
	hwinfo->ospi_fifo_depth = (regval >> 4) & 0x00FF;
	hwinfo->ospi_prescaler_size = (regval >> 12) & 0x00FF;
	hwinfo->ospi_axi_id = (regval >> 20) & 0x000F;
	hwinfo->ospi_mem_mapped = (regval >> 24) & 0x000F;
	hwinfo->ospi_master_mode = (regval >> 28) & 0x000F;
	return OSPI_OP_COMPLETE;
}

ospi_status_t octalspi_set_phase(uint8_t id, ospi_phases_t *phase) {
	ospi_status_t retval = OSPI_OP_REQUEST;

	if (phase == NULL) {
		return retval;
	}

	if((retval = ospi_wait_and_clear(id) != OSPI_OP_COMPLETE))
		return retval;

	switch (phase->type) {
	case OSPI_INSTRUCTION_PHASE:
		ospi_set_instruction_phase(id, phase);
	break;
	case OSPI_ADDRESS_PHASE:
		ospi_set_address_phase(id, phase);
	break;
	case OSPI_ALTERNATE_PHASE:
		ospi_set_alternate_phase(id, phase);
	break;
	case OSPI_DUMMY_PHASE:
		ospi_set_dummy_phase(id, phase);
	break;
	case OSPI_DATA_PHASE:
		ospi_set_data_phase(id, phase);
	break;
	}
	return OSPI_OP_COMPLETE;
}

/**
 * @brief   			Setup Hyperbus latency
 *
 * @param[in] id		OSPI_ID1 to dial with OctalSPI 1 interface, OSPI_ID2 otherwise for second interface.
 *
 * @param[in] latency	Latency to control the initial access and read/write recovery time
 *
 * @return				No error if setup, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_set_hyperbus(uint8_t id, ospi_latency_t *latency) {
	uint32_t regval;

	if ((octalspi_get_status(id) == OSPI_OP_BUSY) || 
		(OSPI_ICLK < 200000000U)) {
			return OSPI_OP_ERROR;
	}

	/* In HyperBusTM mode the device timing and the latency
	 * mode must be 32bits addressed.
	 */
	regval = 0x0U;
	/* Disable OctalSPI */
	octalspi_tags[id]->CR.B.EN = 0x0U;
	if (latency == NULL) {
		/* Use default configuration extracted from NVRC of
		 * Cypress MCP HyperFlash/RAM external device.
		 * Valid factory read default latency [5-16] clocks
		 * @ [52-166]MHz/1.8V or [52-100]MHz/3.3V to have
		 * ~96ns Tacc. Use average[min,max].
		 */
		uint16_t tacc = (octalspi_tags[id]->DCR2.R & 0xFFU) + 0x1;
		tacc = OSPI_ICLK / 1000000U / tacc;
		tacc = 96 * 10 / tacc;  // max microseconds
		tacc = (tacc + 5) / 2; // average[min,max]
		tacc -= 1; // rounded down
		OSPI_SET(regval, OSPI_HB_RWL(0x1U));
		OSPI_SET(regval, OSPI_HB_ATL(tacc));
		OSPI_SET(regval, OSPI_HB_WZL(0x0U));
		OSPI_SET(regval, OSPI_HB_LM(0x1U));
		ospi_drv[id].default_latency = 0x1U;
	} else {
		OSPI_SET(regval, OSPI_HB_RWL(latency->recovery_time));
		OSPI_SET(regval, OSPI_HB_ATL(latency->initial_access));
		OSPI_SET(regval, OSPI_HB_WZL(latency->no_write_latency));
		OSPI_SET(regval, OSPI_HB_LM(latency->fixed_latency));
		ospi_drv[id].default_latency = 0x0U;
	}
	octalspi_tags[id]->HLCR.R = regval;
	/* Enable OctalSPI */
	octalspi_tags[id]->CR.B.EN = 0x1U;
	return OSPI_OP_COMPLETE;
}

/**
 * @brief   			Get current Hyperbus latency values
 *
 * @param[in] id		OSPI_ID1 to dial with OctalSPI 1 interface, OSPI_ID2 otherwise for second interface.
 *
 * @param[out] latency	Get latency currently used for initial access and read/write recovery time
 *
 * @return				No error if setup, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_get_hyperbus(uint8_t id, ospi_latency_t *latency) {
	uint32_t val = octalspi_tags[id]->HLCR.R;
	
	latency->recovery_time = (val & 0xff0000) >> 16;
	latency->initial_access = (val & 0xff00) >> 8;
	latency->no_write_latency = (val & 0x2) >> 1;
	latency->fixed_latency = (val & 0x1);

	return OSPI_OP_COMPLETE;
}

/**
 * @brief   			Set single or dual Chip Select to use.
 *
 * @param[in] id		With id = OSPI_ID1 only the Chip Select 1 will be enabled
 * 						and with OSPI_ID2 both. CSN1 and CSN2.
 *
 * @return				No error if setup, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_set_nocsn(uint8_t id) {
	ospi_status_t retval = OSPI_OP_REQUEST;

	if ((octalspi_get_status(OSPI_ID1) != OSPI_OP_COMPLETE) ||
		(octalspi_get_status(OSPI_ID2) != OSPI_OP_COMPLETE)) {
		return retval;
	}
	/* Only if both OctalSPI's are disabled the IO Manager register
	 * can be changed.
	 */
	if (octalspi_tags[OSPI_ID2] != 0x0U)
		octalspi_tags[OSPI_ID2]->CR.B.EN = 0x0U;
	if (octalspi_tags[OSPI_ID1] != 0x0U)
		octalspi_tags[OSPI_ID1]->CR.B.EN = 0x0U;

	*(vuint32_t *)((uint32_t)OSPI_MEMREG1 + 0x2000U) = id - 0x1U;

	if (octalspi_tags[OSPI_ID1] != 0x0U)
		octalspi_tags[OSPI_ID1]->CR.B.EN = 0x1U;
	if (octalspi_tags[OSPI_ID2] != 0x0U)
		octalspi_tags[OSPI_ID2]->CR.B.EN = 0x1U;

	ospi_drv[0x0U].default_nocsn = id;

	return OSPI_OP_COMPLETE;
}

/**
 * @brief   			Get the number of Chip Select enabled.
 *
 * @param[out] nocsn	is 1 if single-CSN, 2 if in dual-CSN.
 *
 * @return				No error if setup, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_get_nocsn(uint8_t *nocsn) {
	*nocsn = (*(vuint32_t *)((uint32_t)OSPI_MEMREG1 + 0x2000U) + 0x1U);
	return OSPI_OP_COMPLETE;
}

/**
 * @brief   			Put external device in low-power mode.
 *
 * @param[out] timeout	Enter low-power mode after a timeout elapsed. Zero means
 * 						always power-on.
 * @param[in] id		With id = OSPI_ID1 is the HyperRAM to go in stand-by,
 * 						HyperFLASH otherwise. This bit is valid only when in memory-mapped
 * 						mode. Activating this bit causes the chip select (nCS) to be released
 * 						and thus reduce the consumption if there has not been an access after
 * 						a certain amount of time, the timeout value.
 * @return				No error if setup, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_set_lowpower(uint8_t id, uint16_t timeout) {
	if (octalspi_get_status(id) == OSPI_OP_COMPLETE) {
		if (ospi_drv[id].current_mode == OSPI_MEM_MAPPED_MODE) {
			/* This bit is used to release the nCS after
			 * a LPTR timeout value to reduce consumption.
			 */
			if (timeout == 0x0U) {
				octalspi_tags[id]->CR.B.TOIE = 0x0U;
				octalspi_tags[id]->CR.B.TCEN = 0x0U;
				octalspi_tags[id]->LPTR.R = 0x0U;
			} else {
				octalspi_tags[id]->LPTR.R = timeout;
				octalspi_tags[id]->CR.B.TOIE = 0x1U;
				octalspi_tags[id]->CR.B.TCEN = 0x1U;
			}
		return OSPI_OP_COMPLETE;
		}
	}
	return OSPI_OP_REQUEST;
}

/**
 * @brief   			Get the external low-power status mode.
 *
 * @param[out] pmode	is 1 if in low-power mode, 0 otherwise.
 * @param[in] id		With id = OSPI_ID1 is the HyperRAM status,
 * 						HyperFLASH otherwise.
 * @return				No error if setup, error otherwise.
 *
 * @api
 */
ospi_status_t octalspi_get_lowpower(uint8_t id, uint8_t *pmode) {
	if (ospi_isvalid(id) != 0x1U) {
		return OSPI_OP_REQUEST;
	}
	*pmode = octalspi_tags[id]->CR.B.TCEN;
	return OSPI_OP_COMPLETE;
}

ospi_status_t octalspi_set_polling(uint8_t id, ospi_match_t *polling) {
	if (ospi_drv[id].current_mode == OSPI_POLLING_MODE) {
		octalspi_tags[id]->CR.B.PMM = polling->or_mode;
		octalspi_tags[id]->PSMKR.R = polling->mask;
		octalspi_tags[id]->PSMAR.R = polling->match;
		octalspi_tags[id]->PIR.R = 0xAU;
		/* Automatic-polling mode stops as soon as there is a match if 1 is set */
		octalspi_tags[id]->CR.B.APMS = polling->one_mode;
		/* Clear all status flags */
		octalspi_tags[id]->FCR.R = 0x1BU;
		return OSPI_OP_COMPLETE;
	}
	return OSPI_OP_REQUEST;
}

/**
 * @brief   OctalSPI 1 Timeout Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector187) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID1, OSPI_NO_IRQ(OSPI_ID1, OSPI_NO_TO));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 1 Status match Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector188) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID1, OSPI_NO_IRQ(OSPI_ID1, OSPI_NO_SM));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 1 FIFO threshold  Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector189) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID1, OSPI_NO_IRQ(OSPI_ID1, OSPI_NO_FT));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 1 Transfer complete Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector190) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID1, OSPI_NO_IRQ(OSPI_ID1, OSPI_NO_TC));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 1 Transfer error Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector191) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID1, OSPI_NO_IRQ(OSPI_ID1, OSPI_NO_TE));
  IRQ_EPILOGUE();
}

/* OctalSPI 2 interrupts configuration */
#if (OSPI_INSTANCES == 0x2U)
/**
 * @brief   OctalSPI 2 Timeout Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector193) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID2, OSPI_NO_IRQ(OSPI_ID2, OSPI_NO_TO));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 2 Status match Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector194) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID2, OSPI_NO_IRQ(OSPI_ID2, OSPI_NO_SM));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 2 FIFO threshold  Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector195) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID2, OSPI_NO_IRQ(OSPI_ID2, OSPI_NO_FT));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 2 Transfer complete Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector196) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID2, OSPI_NO_IRQ(OSPI_ID2, OSPI_NO_TC));
  IRQ_EPILOGUE();
}
/**
 * @brief   OctalSPI 2 Transfer error Interrupt handler.
 *
 * @isr
 */
IRQ_HANDLER(vector197) {
  IRQ_PROLOGUE();
  ospi_irq_handler(OSPI_ID2, OSPI_NO_IRQ(OSPI_ID2, OSPI_NO_TE));
  IRQ_EPILOGUE();
}
#endif
#endif /* LLD_USE_OCTALSPI */
/** @} */
