/****************************************************************************
*
* Copyright © 2021 STMicroelectronics - All Rights Reserved
*
* This software is licensed under SLA0098 terms that can be found in the
* DM00779817_1_0.pdf file in the licenses directory of this software product.
*
* THIS SOFTWARE IS DISTRIBUTED "AS IS," AND ALL WARRANTIES ARE DISCLAIMED,
* INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*
*****************************************************************************/
/**
 * @file    sdmmc_lld.h
 * @brief   SPC5xx SDMMC control unit.
 *
 * @addtogroup SDMMC
 * @{
 */

#include "spc5_lld.h"
#include "lldconf.h"

#ifndef SDMMC_LLD_H_
#define SDMMC_LLD_H_

/**
 * @brief    SDMMC Card type.
 */
typedef enum {
	SDMMC_TYPE_REMOVABLE,	/**<  Removable card. */
	SDMMC_TYPE_EMBEDDED,	/**<  Socked card. */
	SDMMC_TYPE_SHARED,		/**<  POP card */
} sdmmc_type_t;

/**
 * @brief    SDMMC Card speed configuration.
 */
typedef enum {
	SDMMC_SPEED_LEGACY,		/**< 400KHz-26MHz supported by default */
	SDMMC_SPEED_HIGH,		/**< 26-52MHz could be supported */
	SDMMC_SPEED_HS200,		/**< 52-200MHz could be supported */
} sdmmc_speed_t;

/**
 * @brief    SDMMC Card bus width configuration.
 */
typedef enum {
	SDMMC_BUS_1BIT_SINGLE,		/**< For only 1 data line single rate */
	SDMMC_BUS_4BIT_SINGLE,		/**< For 4 data lines single rate */
	SDMMC_BUS_8BIT_SINGLE,		/**< For 8 data lines single rate */
	SDMMC_BUS_4BIT_DUAL,		/**< For 4 data lines dual rate */
	SDMMC_BUS_8BIT_DUAL,		/**< For 8 data lines dual rate */
} sdmmc_bus_t;

/**
 * @brief    SDMMC Card speed configuration.
 */
typedef enum {
	SDMMC_SWITCH_AWAKE,		/**< Awake in normal state operation */
	SDMMC_SWITCH_SLEEP,		/**< Sleep to reduce power consumption */
} sdmmc_switch_t;

/**
 * @brief    SDMMC Card identification.
 */
typedef struct {
	uint8_t manufactur_id;		/**< Identifies the device manufacturer */
	sdmmc_type_t device_type;	/**< Indicates the device type. */
	uint8_t optional_id;		/**< Optional identify */
	uint8_t product_name[7];	/**< Product name is a string, 6 ASCII characters long. */
	uint8_t product_version;	/**< Product revision, is composed of two Binary Coded Decimal (BCD) digits. */
	uint32_t product_serial_no;	/**< Product serial number, is a 32-bit unsigned binary integer. */
	uint8_t manufactur_data;	/**< Product manufacturing date. */
	uint8_t dual_voltage;		/**< Dual voltage support. */
} sdmmc_card_id_t;

/**
 * @brief    SDMMC Card operation.
 */
typedef struct {
	uint8_t  device_version;	/**< Device specification version */
	uint32_t device_size;		/**< The total device size */
	uint32_t device_rca;		/**< The relative device access identy */
	uint32_t max_speed;			/**< The max speed supported. */
	uint16_t command_class;		/**< The command class supported */
	uint16_t read_block_length;	/**< The basic read block size */
	uint16_t write_block_length;/**< The basic write block size */
	uint16_t erase_size;		/**< The basic erase block size */
	uint8_t  bus_width;			/**< The current bus width */
	uint8_t  power_class;		/**< The power class supported */
	uint8_t	 cmd6_timeout;		/**< The SWITCH command timeout  */
	uint8_t	 data_timeout;		/**< The DATA command timeout  */
	uint16_t sa_timeout;		/**< The Sleep/Awake timeout  */
	uint8_t  hpi_support;		/**< High priority support for stop transmission  */
} sdmmc_card_op_t;

/**
 * @brief    SDMMC Card operation result.
 */
typedef enum {
	SDMMC_NO_ERROR,			/**< Card request satisfied.. */
	SDMMC_CARD_ERROR,		/**< Card discovering error. */
	SDMMC_TIMEOUT_ERROR,	/**< Command timeout error. */
	SDMMC_CMD_ERROR,		/**< Card command error. */
	SDMMC_CLOCK_ERROR,		/**< Clock unstable or setup error. */
	SDMMC_BOOT_ERROR,		/**< Boot sequence error. */
	SDMMC_SM_ERROR,			/**< Finite state machine error. */
	SDMMC_REQUEST_ERROR,	/**< Request or feature unsupported. */
	SDMMC_FATAL_ERROR,		/**< Fatal internal error. Reinit is required. */
	SDMMC_IDENTIFY_ERROR,	/**< Card identification error. */
	SDMMC_OPERATION_ERROR,	/**< Card operation error. */
} sdmmc_result_t;

sdmmc_result_t sdmmc_init(void);
sdmmc_card_id_t *sdmmc_get_identification(void);
sdmmc_card_op_t *sdmmc_get_operation(void);
uint16_t sdmmc_get_read_size(void);
uint16_t sdmmc_get_write_size(void);
uint16_t sdmmc_get_erase_size(void);
sdmmc_result_t sdmmc_write(uint32_t *src, uint32_t *dst, uint32_t nbytes);
sdmmc_result_t sdmmc_read(uint32_t *src, uint32_t *dst, uint32_t nbytes);
sdmmc_result_t sdmmc_switching_high_speed(sdmmc_speed_t speed);
sdmmc_result_t sdmmc_changing_data_bus(sdmmc_bus_t bus);
sdmmc_result_t sdmmc_erase(uint32_t *dst, uint32_t nbytes);
sdmmc_result_t sdmmc_switching_power(sdmmc_switch_t mode);
sdmmc_result_t sdmmc_deinit(void);
#endif /* SDMMC_LLD_H_ */
/** @} */
