/****************************************************************************
*
* Copyright © 2019 STMicroelectronics - All Rights Reserved
*
* This software is licensed under SLA0098 terms that can be found in the
* DM00779817_1_0.pdf file in the licenses directory of this software product.
* 
* THIS SOFTWARE IS DISTRIBUTED "AS IS," AND ALL WARRANTIES ARE DISCLAIMED, 
* INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*
*****************************************************************************/
/**
 * @file    octalspi_lld.h
 * @brief   SPC5xx OctalSPI control unit.
 *
 * @addtogroup OctalSPI
 * @{
 */
#ifndef _OCTALSPI_LLD_H_
#define _OCTALSPI_LLD_H_

#include "spc5_lld.h"
#include "lldconf.h"

#if (LLD_USE_OCTALSPI == TRUE) || defined(__DOXYGEN__)
#include "octalspi_lld_cfg.h"

/**
 * @brief    OcotalSPI Device ID.
 */
#define OSPI_ID1	1U 		/**< @brief   OctalSPI ID for HyperRAM base address request */
#define OSPI_ID2	2U 		/**< @brief   OctalSPI ID for HyperFLASH base address request */

/**
 * @brief    OcotalSPI operation status.
 */
typedef enum {
	OSPI_OP_COMPLETE = 0x00U,	/**< @brief   Operation complete successfull. Any modes. */
	OSPI_OP_FIFOFULL,			/**< @brief   FIFO full. Any modes. */
	OSPI_OP_BUSY,				/**< @brief   Operation is ongoing. Any modes.*/
	OSPI_OP_TIMEOUT,			/**< @brief   Timeout expired. Any modes. */
	OSPI_OP_MATCHDATA,			/**< @brief   Staus match data. Polling modes. */
	OSPI_OP_FIFOTHRESH,			/**< @brief   FIFO threshold reached. Indirect mode. */
	OSPI_OP_ERROR,				/**< @brief   Invalid address access. Indirect mode.*/
	OSPI_OP_ABORT,				/**< @brief   Transfer aborted. Any modes except indirect mode.*/
	OSPI_OP_REQUEST,			/**< @brief   Invalid operation request or bad parameter. */
} ospi_status_t;

/**
 * @brief    OcotalSPI frame formats.
 */
typedef enum {
	OSPI_REGULARCMD_FRAME = 0x10U,	/**< @brief   OctalSPI communication via classical commands sequences. */
	OSPI_HYPERBUS_FRAME,			/**< @brief   OctalSPI comumnication via HyperBus interface. */
} ospi_formats_t;

/**
 * @brief    OcotalSPI functional modes.
 */
typedef enum {
	OSPI_INDIRECT_MODE = 0x21U,	/**< @brief   OctalSPI operations done via registers.*/
	OSPI_POLLING_MODE,			/**< @brief   OctalSPI status polling on the external device. */
	OSPI_MEM_MAPPED_MODE,		/**< @brief   OctalSPI view as an internal memory. */
} ospi_modes_t;

/**
 * @brief    OcotalSPI signal modes.
 */
typedef enum {
	OSPI_SINGLE_SIGNAL = 0x30U,	/**< @brief   OctalSPI single line to be sent/received serially. */
	OSPI_DUAL_SIGNAL,			/**< @brief   OctalSPI two lines are sent/received simultaneously. */
	OSPI_QUAD_SIGNAL,			/**< @brief   OctalSPI four lines are sent/received simultaneously. */
	OSPI_OCTO_SIGNAL,			/**< @brief   OctalSPI eight lines are sent/received simultaneously. */
} ospi_signals_t;

/**
 * @brief    OcotalSPI phase size.
 */
typedef enum {
	OSPI_8BIT_SIZE = 0x40U,	/**< @brief   OctalSPI 8-bits phase size. */
	OSPI_16BIT_SIZE,		/**< @brief   OctalSPI 16-bits phase size. */
	OSPI_24BIT_SIZE,		/**< @brief   OctalSPI 24-bits phase size. */
	OSPI_32BIT_SIZE,		/**< @brief   OctalSPI 32-bits phase size. */
} ospi_size_t;

/**
 * @brief    OcotalSPI phase modes.
 */
typedef enum {
	OSPI_INSTRUCTION_PHASE = 0x50U,	/**< @brief   OctalSPI set instruction phase. */
	OSPI_ADDRESS_PHASE,				/**< @brief   OctalSPI set address phase. */
	OSPI_ALTERNATE_PHASE,			/**< @brief   OctalSPI set alternate phase. */
	OSPI_DUMMY_PHASE,				/**< @brief   OctalSPI set dummy phase. */
	OSPI_DATA_PHASE,				/**< @brief   OctalSPI set data phase. */
} ospi_phase_t;

/**
 * @brief    OcotalSPI setup Hyperbus phases.
 */
typedef struct {
	/* Common for all phases */
	ospi_phase_t type;		/**< @brief   OctalSPI phase type */
	ospi_size_t size;		/**< @brief   OctalSPI phase size */
	ospi_signals_t signal;	/**< @brief   OctalSPI phase signal */
	uint8_t enable_dtr;		/**< @brief   OctalSPI enable double transfer rate */
	/* Specific for Instruction phase */
	uint8_t only_once;		/**< @brief   OctalSPI send instruction only once */
	/* Specific for Data phase */
	uint8_t enable_strobe;	/**< @brief   OctalSPI enable data strobe management */
	/* Specific for dummy phase */
	uint8_t dummy_cycles;	/**< @brief   OctalSPI number of dummy cycles */
	uint8_t enable_delay;	/**< @brief   OctalSPI enable elay hold quarter cycle */
	uint8_t enable_shift;	/**< @brief   OctalSPI enable sample shift */
} ospi_phases_t;

/**
 * @brief    OcotalSPI setup Hyperbus latency.
 */
typedef struct {
	uint8_t recovery_time;		/**< @brief   Hyperbus recovery time in number of clocks */
	uint8_t initial_access;		/**< @brief   Hyperbus initial access in number of clocks */
	uint8_t no_write_latency;	/**< @brief   Hyperbus no write latency if true */
	uint8_t fixed_latency;		/**< @brief   Hyperbus fixed latency if true*/
} ospi_latency_t;

/**
 * @brief    OcotalSPI setup status polling mode.
 */
typedef struct {
	uint32_t mask;			/**< @brief   The bits mask to consider for matching */
	uint32_t match;			/**< @brief   The data value to match. */
	uint8_t  or_mode;		/**< @brief   OR match mode actived. */
	uint8_t  one_mode;		/**< @brief   Automatic-polling stops after the first match. */
} ospi_match_t;

/**
 * @brief    OctalSPI Hardware information setup.
 */
typedef struct {
	uint32_t ospi_identification;
	uint32_t ospi_magic_id;
	uint8_t  ospi_version;
	uint8_t  ospi_master_mode;
	uint8_t  ospi_mem_mapped;
	uint8_t  ospi_axi_id;
	uint8_t  ospi_prescaler_size;
	uint8_t  ospi_fifo_depth;
	uint8_t  ospi_axi_interface;
} ospi_hwinfo_t;

#ifdef __cplusplus
extern "C" {
#endif
ospi_status_t octalspi_init(void);
uint32_t octalspi_get_baseaddr(uint8_t id);
ospi_status_t octalspi_get_status(uint8_t id);
ospi_status_t octalspi_set_fmode(uint8_t id, ospi_modes_t fmode);
ospi_status_t octalspi_get_fmode(uint8_t id, ospi_modes_t *fmode);
ospi_status_t octalspi_set_phase(uint8_t id, ospi_phases_t *phase);
ospi_status_t octalspi_get_hwinfo(uint8_t id, ospi_hwinfo_t *hwinfo);
ospi_status_t octalspi_set_hyperbus(uint8_t id, ospi_latency_t *latency);
ospi_status_t octalspi_get_hyperbus(uint8_t id, ospi_latency_t *latency);
ospi_status_t octalspi_set_nocsn(uint8_t id);
ospi_status_t octalspi_get_nocsn(uint8_t *nocsn);
ospi_status_t octalspi_set_lowpower(uint8_t id, uint16_t timeout);
ospi_status_t octalspi_get_lowpower(uint8_t id, uint8_t *pmode);
ospi_status_t octalspi_set_polling(uint8_t id, ospi_match_t *polling);
ospi_status_t octalspi_write(uint32_t *src, uint32_t *dst, uint32_t nbytes);
ospi_status_t octalspi_read(uint32_t *src, uint32_t *dst, uint32_t nbytes);
void octalspi_deinit(void);

#ifdef __cplusplus
}
#endif

#endif /* LLD_USE_OCTALSPI */
#endif /* _OCTALSPI_LLD_H_ */
/** @} */
